<?php

namespace Drupal\entity_io\Service;

use Drupal\Core\Database\Connection;

/**
 * Service to manage JSON entity metadata storage.
 */
class JsonStorageService {

  /**
   * The database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * Get the existing record or save a new one and return it.
   *
   * @return array
   *   The row from the entity_io table.
   */
  public function getOrSave(
    string $jsonUuid,
    string $jsonId,
    string $jsonEntityType,
    string $jsonBundle,
    int $entityId,
    string $entityUuid,
    string $entityType,
    string $entityBundle,
  ): array {

    $row = $this->database->select('entity_io_storage', 'js')
      ->fields('js')
      ->condition('json_uuid', $jsonUuid)
      ->range(0, 1)
      ->execute()
      ->fetchAssoc();

    if ($row) {
      return $row;
    }

    $this->database->insert('entity_io_storage')
      ->fields([
        'json_uuid' => $jsonUuid,
        'json_id' => $jsonId,
        'json_entity_type' => $jsonEntityType,
        'json_bundle' => $jsonBundle,
        'entity_id' => $entityId,
        'entity_uuid' => $entityUuid,
        'entity_type' => $entityType,
        'entity_bundle' => $entityBundle,
      ])
      ->execute();

    // Re-fetch the inserted row to return it.
    return $this->database->select('entity_io_storage', 'js')
      ->fields('js')
      ->condition('json_uuid', $jsonUuid)
      ->range(0, 1)
      ->execute()
      ->fetchAssoc();
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $uuid) {
    $row = $this->database->select('entity_io_storage', 'js')
      ->fields('js')
      ->condition('json_uuid', $uuid)
      ->range(0, 1)
      ->execute()
      ->fetchAssoc();

    if ($row) {
      return $row;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntity($id, $entity_type) {
    $row = $this->database->select('entity_io_storage', 'js')
      ->fields('js')
      ->condition('json_id', $id)
      ->condition('json_entity_type', $entity_type)
      ->range(0, 1)
      ->execute()
      ->fetchAssoc();

    if ($row) {
      return $row;
    }
    return NULL;
  }

}
