<?php

declare(strict_types=1);

namespace Drupal\entity_language_access\EventSubscriber;

use Drupal\Core\EventSubscriber\CustomPageExceptionHtmlSubscriber;
use Drupal\entity_language_access\Access\EntityLanguageAccess;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;

/**
 * Redirects to fallback content on 403, if configured.
 */
class FallbackContent extends CustomPageExceptionHtmlSubscriber {

  /**
   * {@inheritdoc}
   */
  protected static function getPriority() {
    // The priority of Core's CustomPageExceptionHtmlSubscriber is -50. Lower
    // priority is run earlier. We need to run before Core's subscriber.
    return -49;
  }

  /**
   * {@inheritdoc}
   */
  public function on403(ExceptionEvent $event): void {
    if (!$this->needsRedirect($event)) {
      // This 403 has not been not triggered by Entity Language Access.
      return;
    }
    if (!$this->isFallbackContentEnabled()) {
      // Fallback content instead of general 403 page has not been enabled.
      return;
    }
    $nid = $this->getFallbackContentNodeId();
    if ($nid < 1) {
      // Fallback content node id not set.
      return;
    }
    $this->makeSubrequestToCustomPath($event, '/node/' . $nid, Response::HTTP_FORBIDDEN);
  }

  /**
   * Whether the 403 exception needs a redirect to fallback content.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   Exception event to check.
   *
   * @return bool
   *   Wether the 403 exception needs a redirect to fallback content.
   */
  protected function needsRedirect(ExceptionEvent $event): bool {
    $requirement = $event
      ->getRequest()
      ->attributes
      ->get('_route_object')
      ?->getRequirement('_entity_language_access');
    if ($requirement !== 'true') {
      // The 403 exception was thrown for a route not protected by entity
      // language access.
      return FALSE;
    }
    // Make sure that access was prevented by entity language access by checking
    // the reason of the access result.
    $reason = $event
      ->getRequest()
      ->attributes
      ->get('_access_result')
      ?->getReason();
    return $reason === EntityLanguageAccess::REASON;
  }

  /**
   * Whether fallback content instead of 403 page has been enabled in config.
   *
   * @return bool
   *   Whether fallback content instead of 403 page has been enabled in config.
   */
  protected function isFallbackContentEnabled(): bool {
    $config = $this->configFactory->get('entity_language_access.settings');
    if ($config->isNew()) {
      return FALSE;
    }
    return $config->get('is_fallback_content_enabled');
  }

  /**
   * Gets the node id to use as fallback content.
   *
   * @return int
   *   Fallback content node id. 0, if it doesn't exist.
   */
  protected function getFallbackContentNodeId(): int {
    $config = $this->configFactory->get('entity_language_access.settings');
    if ($config->isNew()) {
      return 0;
    }
    return $config->get('fallback_content_nid');
  }

}
