<?php

declare(strict_types=1);

namespace Drupal\entity_language_access\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an administrative settings form for Entity Language Access module.
 */
class AdminSettingsForm extends ConfigFormBase {

  use RedundantEditableConfigNamesTrait;

  /**
   * Constructs a new AdminSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-return self
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_language_access_admin_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-param array<string|int,mixed> $form
   * @phpstan-return array<string|int,mixed>
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['fallback_content'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Fallback Content'),
      '#tree' => FALSE,
    ];

    $form['fallback_content']['is_fallback_content_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display fallback content for missing translations'),
      '#config_target' => 'entity_language_access.settings:is_fallback_content_enabled',
      '#description' => $this->t('Check this box, if you want to show a specific content instead of a general 403 error page in case of missing translations.'),
    ];

    $form['fallback_content']['fallback_content_nid'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'node',
      '#selection_settings' => [
        'target_bundles' => ['page'],
      ],
      '#title' => $this->t('Content to display for untranslated entities'),
      '#description' => $this->t('Content that should be shown, if an entity is missing translations. Note: The selected content must be translated to all available languages and be accessible to all users.'),
      '#config_target' => new ConfigTarget(
        'entity_language_access.settings',
        'fallback_content_nid',
        fn($value) => $this->entityTypeManager->getStorage('node')->load($value) ?? NULL,
      ),
      '#required' => TRUE,
      '#states'   => [
        'visible' => [
          ':input[name="is_fallback_content_enabled"]' => [
            'checked' => TRUE,
          ],
        ],
        'required' => [
          ':input[name="is_fallback_content_enabled"]' => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

}
