<?php

declare(strict_types=1);

namespace Drupal\entity_language_access\Routing;

use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Symfony\Component\Routing\RouteCollection;

/**
 * Alters access checks of canonical entity routes.
 */
class RouteSubscriber extends RouteSubscriberBase {

  /**
   * Constructs a new RouteSubscriber object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection): void {
    foreach ($this->getRouteNamesToAlter() as $route_name) {
      if ($route = $collection->get($route_name)) {
        $route->setRequirement('_entity_language_access', 'true');
      }
    }
  }

  /**
   * Gets route names to apply access check to.
   *
   * @return string[]
   *   Route names to apply access check to.
   */
  protected function getRouteNamesToAlter(): array {
    $route_names = [];
    $definitions = $this->entityTypeManager->getDefinitions();
    foreach ($definitions as $entity_type) {
      if (!$entity_type instanceof ContentEntityTypeInterface) {
        // Not a content entity type.
        continue;
      }
      if (!$entity_type->hasLinkTemplate('canonical')) {
        // Has no canonical route.
        continue;
      }
      if (!$entity_type->isTranslatable()) {
        // Doesn't support translation.
        continue;
      }
      $route_names[] = 'entity.' . $entity_type->id() . '.canonical';
    }
    return $route_names;
  }

}
