<?php

declare(strict_types=1);

namespace Drupal\entity_language_access\Access;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatch;
use Drupal\Core\TypedData\TranslatableInterface;
use Symfony\Component\Routing\Route;

/**
 * Checks access to content entities based on available translations.
 */
class EntityLanguageAccess implements AccessInterface {

  /**
   * The reason to set on forbidden access results.
   *
   * @var string
   */
  const REASON = 'Entity has no translation for current language.';

  /**
   * Constructs a new EntityLanguageAccess object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   */
  public function __construct(
    private LanguageManagerInterface $languageManager,
  ) {
  }

  /**
   * Checks access.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Run access checks for this account.
   * @param \Symfony\Component\Routing\Route $route
   *   Run access checks for this route.
   * @param \Drupal\Core\Routing\RouteMatch $route_match
   *   Run access checks for this route match.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(AccountInterface $account, Route $route, RouteMatch $route_match) {
    if ($account->hasPermission('bypass entity_language_access')) {
      return AccessResult::allowed()->cachePerPermissions();
    }
    $route_name = $route_match->getRouteName();
    $parameters = $route_match->getParameters();
    $entity = NULL;
    foreach ($parameters as $parameter) {
      if ($parameter instanceof ContentEntityInterface) {
        if (!$this->isEntityTranslatable($parameter)) {
          continue;
        }
        $canonical = 'entity.' . $parameter->getEntityTypeId() . '.canonical';
        if ($route_name === $canonical) {
          $entity = $parameter;
          break;
        }
      }
    }
    if (!$entity) {
      return AccessResult::allowed()->cachePerPermissions();
    }
    $current_language = $this->languageManager->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();
    if ($current_language !== $entity->language()->getId()) {
      return AccessResult::forbidden()
        ->cachePerPermissions()
        ->addCacheableDependency($entity)
        // @note The access result reason is checked elsewhere in the module to
        // easily identify blocked access due to missing translations without
        // rerunning all the checks.
        ->setReason(self::REASON);
    }
    return AccessResult::allowed()
      ->cachePerPermissions()
      ->addCacheableDependency($entity);
  }

  /**
   * Checks, if an entity is translatable.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE, if entity is translatable. Otherwise FALSE.
   */
  protected function isEntityTranslatable(ContentEntityInterface $entity): bool {
    return $entity instanceof TranslatableInterface && $entity->isTranslatable();
  }

}
