<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_language_access\Functional;

use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests Entity Language Access.
 *
 * @group entity_language_access
 */
class EntityLanguageAccessTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_translation',
    'entity_language_access',
    'node',
  ];

  /**
   * Users created during set-up.
   *
   * @var \Drupal\user\Entity\User[]
   */
  protected $users;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->users['admin_user'] = $this->drupalCreateUser([
      'administer languages',
      'access administration pages',
      'administer entity_language_access',
    ]);

    // Create an article node type.
    \Drupal::entityTypeManager()->getStorage('node_type')
      ->create([
        'type' => 'article',
        'name' => 'Article',
      ])
      ->save();

    // Create a page node type.
    \Drupal::entityTypeManager()->getStorage('node_type')
      ->create([
        'type' => 'page',
        'name' => 'Page',
      ])
      ->save();

    // Add a second language.
    ConfigurableLanguage::createFromLangcode('fr')->save();

    // Enable content translation for node pages.
    \Drupal::service('content_translation.manager')->setEnabled('node', 'page', TRUE);
  }

  /**
   * Test Entity Language Access.
   */
  public function testEntityLanguageAccess(): void {
    // Verify front page is accessible.
    $this->drupalGet('');
    $this->assertSession()->statusCodeEquals(200);

    // Create a published node.
    $node = \Drupal::entityTypeManager()->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => $this->randomMachineName(),
        'status' => 1,
      ]);
    $node->save();

    // Verify node is accessible in original language.
    $this->drupalGet('/node/' . $node->id());
    $this->assertSession()->statusCodeEquals(200);

    // Verify node is not accessible in translated language, because there is
    // no translation in the current language.
    $this->drupalGet('/fr/node/' . $node->id());
    $this->assertSession()->statusCodeEquals(403);

    // Add translation.
    $translation = $node->addTranslation('fr', $node->toArray());
    $node->save();

    // Verify node is accessible in translated language, because there is a
    // translation in the current language.
    $this->drupalGet('/fr/node/' . $node->id());
    $this->assertSession()->statusCodeEquals(200);

    // Add another node.
    $node2 = \Drupal::entityTypeManager()->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => $this->randomMachineName(),
        'status' => 1,
      ]);
    $node2->save();

    // Verify node is accessible in original language.
    $this->drupalGet('/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(200);

    // Verify node is not accessible in translated language, because there is
    // no translation in the current language.
    $this->drupalGet('/fr/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(403);

    $bypass_user = $this->drupalCreateUser([
      'bypass entity_language_access',
    ]);
    $this->drupalLogin($bypass_user);

    // Verify node is accessible in original language.
    $this->drupalGet('/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(200);

    // Verify node is accessible in translated language, because the logged in
    // user has the bypass permission.
    $this->drupalGet('/fr/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(200);

    // Verify settings page is accessible.
    $admin_user = $this->drupalCreateUser([
      'access administration pages',
      'administer entity_language_access',
    ]);
    $this->drupalLogin($admin_user);
    $this->drupalGet('admin/config/regional/entity_language_access');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Entity Language Access');
    $this->assertSession()->pageTextContains('Fallback Content');

    // Configure a fallback content.
    $edit = [
      'is_fallback_content_enabled' => '1',
      'fallback_content_nid' => $node->label() . ' (' . $node->id() . ')',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->statusMessageContains('The configuration options have been saved.');
    $this->assertSession()->statusCodeEquals(200);

    // Verify node is accessible in original language.
    $this->drupalGet('/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(200);

    // Verify node is not accessible in translated language and fallback content
    // is shown instead.
    $this->drupalGet('/fr/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(403);
    $this->assertSession()->pageTextContains($node->label());

    // Create a published article node.
    $node_article = \Drupal::entityTypeManager()->getStorage('node')
      ->create([
        'type' => 'article',
        'title' => $this->randomMachineName(),
        'status' => 1,
      ]);
    $node_article->save();

    // Verify article node is accessible in original language.
    $this->drupalGet('/node/' . $node_article->id());
    $this->assertSession()->statusCodeEquals(200);

    // Verify article node is accessible in translated language without having
    // a translation, because articles have not been marked as translatable.
    $this->drupalGet('/fr/node/' . $node_article->id());
    $this->assertSession()->statusCodeEquals(200);

    // Unpublish second node.
    $node2->setUnpublished()->save();

    // Verify node is not accessible in original language.
    $this->drupalGet('/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(403);

    // Verify node is not accessible in translated language.
    $this->drupalGet('/fr/node/' . $node2->id());
    $this->assertSession()->statusCodeEquals(403);
  }

}
