<?php

namespace Drupal\entity_lifecycle\Plugin\LifecycleCondition;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Condition based on content age (time since creation/modification).
 *
 * @LifecycleCondition(
 *   id = "age",
 *   label = @Translation("Content age"),
 *   description = @Translation("Content older than X months and/or unchanged for Y months."),
 *   category = "content",
 *   weight = 0
 * )
 */
class AgeCondition extends LifecycleConditionBase {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->time = $container->get('datetime.time');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'created_months' => 0,
      'changed_months' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    $created_threshold = (int) ($this->conditionConfiguration['created_months'] ?? 0);
    $changed_threshold = (int) ($this->conditionConfiguration['changed_months'] ?? 0);

    // If no thresholds set, condition cannot be evaluated meaningfully.
    if ($created_threshold <= 0 && $changed_threshold <= 0) {
      return FALSE;
    }

    $now = $this->time->getRequestTime();
    $seconds_per_month = 30 * 24 * 60 * 60;

    // Check created date condition (if set).
    if ($created_threshold > 0) {
      if (!$entity->hasField('created')) {
        return FALSE;
      }
      $created = (int) $entity->get('created')->value;
      $age_months = ($now - $created) / $seconds_per_month;
      if ($age_months < $created_threshold) {
        return FALSE;
      }
    }

    // Check changed date condition (if set).
    if ($changed_threshold > 0) {
      if (!$entity->hasField('changed')) {
        return FALSE;
      }
      $changed = (int) $entity->get('changed')->value;
      $unchanged_months = ($now - $changed) / $seconds_per_month;
      if ($unchanged_months < $changed_threshold) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    $now = $this->time->getRequestTime();
    $seconds_per_month = 30 * 24 * 60 * 60;

    $values = [];

    if ($entity->hasField('created')) {
      $created = (int) $entity->get('created')->value;
      $values['age'] = ($now - $created) / $seconds_per_month;
    }

    if ($entity->hasField('changed')) {
      $changed = (int) $entity->get('changed')->value;
      $values['unchanged'] = ($now - $changed) / $seconds_per_month;
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    $values = $this->getValue($entity);
    $parts = [];

    if (isset($values['age'])) {
      $parts[] = $this->t('@months months old', ['@months' => round($values['age'], 1)]);
    }
    if (isset($values['unchanged'])) {
      $parts[] = $this->t('unchanged @months months', ['@months' => round($values['unchanged'], 1)]);
    }

    return implode(', ', $parts);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['created_months'] = [
      '#type' => 'number',
      '#title' => $this->t('Created more than'),
      '#default_value' => $configuration['created_months'] ?? 0,
      '#min' => 0,
      '#max' => 120,
      '#field_suffix' => $this->t('months ago'),
      '#size' => 5,
    ];

    $form['changed_months'] = [
      '#type' => 'number',
      '#title' => $this->t('Last updated more than'),
      '#default_value' => $configuration['changed_months'] ?? 0,
      '#min' => 0,
      '#max' => 120,
      '#field_suffix' => $this->t('months ago'),
      '#size' => 5,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $created = (int) ($this->conditionConfiguration['created_months'] ?? 0);
    $changed = (int) ($this->conditionConfiguration['changed_months'] ?? 0);

    $parts = [];
    if ($created > 0) {
      $parts[] = (string) $this->t('age >= @months months', ['@months' => $created]);
    }
    if ($changed > 0) {
      $parts[] = (string) $this->t('unchanged >= @months months', ['@months' => $changed]);
    }

    return implode(' AND ', $parts);
  }

}
