<?php

namespace Drupal\entity_lifecycle\Plugin\LifecycleCondition;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;

/**
 * Default condition that always evaluates to TRUE.
 *
 * Use this as a catch-all for default/current status. Place it with low
 * weight so it only applies to content that doesn't match other conditions.
 *
 * @LifecycleCondition(
 *   id = "default_status",
 *   label = @Translation("Default status (catch-all)"),
 *   description = @Translation("Always evaluates to TRUE. Use as catch-all for content that doesn't match other conditions."),
 *   category = "general",
 *   weight = 100
 * )
 */
class DefaultStatusCondition extends LifecycleConditionBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    // Always return TRUE - this is a catch-all condition.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    return $this->t('Applies to all content');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    // No configuration needed for catch-all condition.
    $form['info'] = [
      '#markup' => $this->t('This condition always evaluates to TRUE and requires no configuration. Use it as a catch-all for content that does not match other conditions.'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    return $this->t('Catch-all (always applies)');
  }

}
