<?php

namespace Drupal\entity_lifecycle\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionManager;
use Psr\Log\LoggerInterface;

/**
 * Service for scanning entities and marking outdated ones.
 */
class LifecycleScanner {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The date formatter.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * The time service.
   */
  protected TimeInterface $time;

  /**
   * The lifecycle condition manager.
   */
  protected LifecycleConditionManager $conditionManager;

  /**
   * The entity type resolver.
   */
  protected EntityTypeResolver $entityTypeResolver;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Constructs a LifecycleScanner object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    ConfigFactoryInterface $config_factory,
    DateFormatterInterface $date_formatter,
    LoggerChannelFactoryInterface $logger_factory,
    TimeInterface $time,
    LifecycleConditionManager $condition_manager,
    EntityTypeResolver $entity_type_resolver,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->configFactory = $config_factory;
    $this->dateFormatter = $date_formatter;
    $this->logger = $logger_factory->get('entity_lifecycle');
    $this->time = $time;
    $this->conditionManager = $condition_manager;
    $this->entityTypeResolver = $entity_type_resolver;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Scans for outdated entities and marks them with appropriate status.
   *
   * @param bool $dry_run
   *   If TRUE, don't actually update entities.
   * @param string|null $entity_type
   *   Limit to specific entity type.
   * @param string|null $bundle
   *   Limit to specific bundle (not applicable for bundleless types).
   *
   * @return array
   *   Results array with count and entity details.
   */
  public function scanAndMark(bool $dry_run = FALSE, ?string $entity_type = NULL, ?string $bundle = NULL): array {
    $results = [
      'count' => 0,
      'entities' => [],
    ];

    // Process bundled entity types.
    $enabled_bundles = $this->getAllEnabledBundles();

    // Filter by entity type if specified.
    if ($entity_type) {
      $enabled_bundles = isset($enabled_bundles[$entity_type])
        ? [$entity_type => $enabled_bundles[$entity_type]]
        : [];
    }

    foreach ($enabled_bundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $settings) {
        // Skip if filtering by bundle and this isn't it.
        if ($bundle && $bundle_id !== $bundle) {
          continue;
        }

        // Skip if no conditions configured.
        if (empty($settings['conditions'])) {
          continue;
        }

        $entities_to_update = $this->findEntitiesForStatusUpdate(
          $entity_type_id,
          $bundle_id,
          $settings['conditions'],
        );

        foreach ($entities_to_update as $entity_data) {
          if (!$dry_run) {
            $this->updateLifecycleStatus(
              $entity_type_id,
              $entity_data['id'],
              $entity_data['new_status'],
              $entity_data['matched_conditions'] ?? NULL
            );
            $this->logger->notice('Marked @type @id as @status: @label', [
              '@type' => $entity_type_id,
              '@id' => $entity_data['id'],
              '@status' => $entity_data['new_status'],
              '@label' => $entity_data['label'],
            ]);
          }

          $results['entities'][] = $entity_data;
          $results['count']++;
        }
      }
    }

    // Process bundleless entity types.
    $bundleless_results = $this->scanBundlelessEntities($dry_run, $entity_type);
    $results['count'] += $bundleless_results['count'];
    $results['entities'] = array_merge($results['entities'], $bundleless_results['entities']);

    if (!$dry_run && $results['count'] > 0) {
      $this->logger->info('Entity lifecycle scan completed. @count entities updated.', [
        '@count' => $results['count'],
      ]);
    }

    return $results;
  }

  /**
   * Rebuilds lifecycle status by clearing existing statuses and re-scanning.
   *
   * This clears all lifecycle status fields for the specified entity
   * type/bundle and then performs a fresh scan to re-evaluate all conditions.
   *
   * @param bool $dry_run
   *   If TRUE, don't actually update entities.
   * @param string|null $entity_type
   *   Limit to specific entity type.
   * @param string|null $bundle
   *   Limit to specific bundle (not applicable for bundleless types).
   *
   * @return array
   *   Results array with count and entity details.
   */
  public function rebuildAndMark(bool $dry_run = FALSE, ?string $entity_type = NULL, ?string $bundle = NULL): array {
    // First, clear existing lifecycle statuses.
    if (!$dry_run) {
      $this->clearLifecycleStatuses($entity_type, $bundle);
    }

    // Then perform a fresh scan.
    return $this->scanAndMark($dry_run, $entity_type, $bundle);
  }

  /**
   * Clears lifecycle status fields for entities.
   *
   * @param string|null $entity_type
   *   Limit to specific entity type.
   * @param string|null $bundle
   *   Limit to specific bundle (not applicable for bundleless types).
   */
  protected function clearLifecycleStatuses(?string $entity_type = NULL, ?string $bundle = NULL): void {
    // Clear for bundled entity types.
    $enabled_bundles = $this->getAllEnabledBundles();

    // Filter by entity type if specified.
    if ($entity_type) {
      $enabled_bundles = isset($enabled_bundles[$entity_type])
        ? [$entity_type => $enabled_bundles[$entity_type]]
        : [];
    }

    foreach ($enabled_bundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $settings) {
        // Skip if filtering by bundle and this isn't it.
        if ($bundle && $bundle_id !== $bundle) {
          continue;
        }

        $this->clearLifecycleStatusForBundle($entity_type_id, $bundle_id);
      }
    }

    // Clear for bundleless entity types.
    if (!$bundle) {
      $enabled_bundleless = $this->getAllEnabledBundlelessEntityTypes();

      // Filter by entity type if specified.
      if ($entity_type) {
        $enabled_bundleless = isset($enabled_bundleless[$entity_type])
          ? [$entity_type => $enabled_bundleless[$entity_type]]
          : [];
      }

      foreach ($enabled_bundleless as $entity_type_id => $settings) {
        $this->clearLifecycleStatusForBundleless($entity_type_id);
      }
    }
  }

  /**
   * Clears lifecycle status for a specific bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   */
  protected function clearLifecycleStatusForBundle(string $entity_type_id, string $bundle_id): void {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $data_table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();
    $bundle_key = $entity_type->getKey('bundle');

    if (!$bundle_key) {
      return;
    }

    // Get entity IDs that will be affected before clearing.
    $id_key = $entity_type->getKey('id');
    $affected_ids = $this->database->select($data_table, 'e')
      ->fields('e', [$id_key])
      ->condition($bundle_key, $bundle_id)
      ->execute()
      ->fetchCol();

    $this->database->update($data_table)
      ->fields([
        'lifecycle_status' => NULL,
        'lifecycle_condition_details' => NULL,
      ])
      ->condition($bundle_key, $bundle_id)
      ->execute();

    // Clear entity cache only for affected entities.
    if (!empty($affected_ids)) {
      $this->entityTypeManager->getStorage($entity_type_id)->resetCache($affected_ids);
    }

    $this->logger->notice('Cleared lifecycle status for @type:@bundle', [
      '@type' => $entity_type_id,
      '@bundle' => $bundle_id,
    ]);
  }

  /**
   * Clears lifecycle status for a bundleless entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   */
  protected function clearLifecycleStatusForBundleless(string $entity_type_id): void {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $data_table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();
    $id_key = $entity_type->getKey('id');

    // Get entity IDs that will be affected before clearing.
    $affected_ids = $this->database->select($data_table, 'e')
      ->fields('e', [$id_key])
      ->execute()
      ->fetchCol();

    $this->database->update($data_table)
      ->fields([
        'lifecycle_status' => NULL,
        'lifecycle_condition_details' => NULL,
      ])
      ->execute();

    // Clear entity cache only for affected entities.
    if (!empty($affected_ids)) {
      $this->entityTypeManager->getStorage($entity_type_id)->resetCache($affected_ids);
    }

    $this->logger->notice('Cleared lifecycle status for @type', [
      '@type' => $entity_type_id,
    ]);
  }

  /**
   * Scans bundleless entity types for lifecycle status updates.
   *
   * @param bool $dry_run
   *   If TRUE, don't actually update entities.
   * @param string|null $entity_type
   *   Limit to specific entity type.
   *
   * @return array
   *   Results array with count and entity details.
   */
  protected function scanBundlelessEntities(bool $dry_run = FALSE, ?string $entity_type = NULL): array {
    $results = [
      'count' => 0,
      'entities' => [],
    ];

    $enabled_bundleless = $this->getAllEnabledBundlelessEntityTypes();

    if (empty($enabled_bundleless)) {
      return $results;
    }

    // Filter by entity type if specified.
    if ($entity_type) {
      $enabled_bundleless = isset($enabled_bundleless[$entity_type])
        ? [$entity_type => $enabled_bundleless[$entity_type]]
        : [];
    }

    foreach ($enabled_bundleless as $entity_type_id => $settings) {
      // Skip if no conditions configured.
      if (empty($settings['conditions'])) {
        continue;
      }

      $entities_to_update = $this->findBundlelessEntitiesForStatusUpdate(
        $entity_type_id,
        $settings['conditions'],
      );

      foreach ($entities_to_update as $entity_data) {
        if (!$dry_run) {
          $this->updateLifecycleStatus(
            $entity_type_id,
            $entity_data['id'],
            $entity_data['new_status'],
            $entity_data['matched_conditions'] ?? NULL
          );
          $this->logger->notice('Marked @type @id as @status: @label', [
            '@type' => $entity_type_id,
            '@id' => $entity_data['id'],
            '@status' => $entity_data['new_status'],
            '@label' => $entity_data['label'],
          ]);
        }

        $results['entities'][] = $entity_data;
        $results['count']++;
      }
    }

    return $results;
  }

  /**
   * Finds entities that need a status update based on conditions.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array $conditions
   *   Array of condition configurations.
   *
   * @return array
   *   Array of entity data with new status to assign.
   */
  protected function findEntitiesForStatusUpdate(
    string $entity_type_id,
    string $bundle_id,
    array $conditions,
  ): array {
    $results = [];

    // Load entities via the Entity API to properly evaluate conditions.
    $storage = $this->entityTypeManager->getStorage($entity_type_id);

    // Get entity type definition.
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    // Build query for candidate entities.
    $query = $storage->getQuery()
      ->accessCheck(FALSE);

    // Add bundle condition if entity type has bundles.
    $bundle_key = $entity_type->getKey('bundle');
    if ($bundle_key) {
      $query->condition($bundle_key, $bundle_id);
    }

    // Note: We intentionally do NOT filter by published status here.
    // The published_status condition plugin handles this, allowing
    // lifecycle tracking of both published and unpublished content.
    // Exclude entities explicitly excluded from scanning.
    $exclude_group = $query->orConditionGroup()
      ->notExists('lifecycle_exclude')
      ->condition('lifecycle_exclude', 0);
    $query->condition($exclude_group);

    $entity_ids = $query->execute();

    if (empty($entity_ids)) {
      return $results;
    }

    // Load entities in batches.
    foreach (array_chunk($entity_ids, 50) as $batch_ids) {
      $entities = $storage->loadMultiple($batch_ids);

      foreach ($entities as $entity) {
        if (!$entity instanceof ContentEntityInterface) {
          continue;
        }

        $current_status = $entity->hasField('lifecycle_status')
          ? $entity->get('lifecycle_status')->value
          : NULL;

        // Find the first matching condition.
        $new_status = $this->evaluateConditions($entity, $conditions);

        // Skip if no status applies or status hasn't changed.
        if ($new_status === NULL || $new_status === $current_status) {
          continue;
        }

        // Get entity details for the result.
        $label_key = $entity_type->getKey('label');
        $label = $entity->hasField($label_key) ? $entity->get($label_key)->value : "Entity {$entity->id()}";

        $created = $entity->hasField('created') ? (int) $entity->get('created')->value : 0;
        $changed = $entity->hasField('changed') ? (int) $entity->get('changed')->value : 0;

        $results[] = [
          'entity_type' => $entity_type_id,
          'id' => $entity->id(),
          'label' => $label,
          'bundle' => $bundle_id,
          'created' => $created ? $this->dateFormatter->format($created, 'short') : '-',
          'changed' => $changed ? $this->dateFormatter->format($changed, 'short') : '-',
          'current_status' => $current_status,
          'new_status' => $new_status,
          'matched_conditions' => $this->getMatchedConditionsSummary($entity, $conditions),
        ];
      }
    }

    return $results;
  }

  /**
   * Evaluates conditions against an entity.
   *
   * Supports both the legacy structure (flat plugins array) and the new
   * structure with condition groups and AND/OR operators between groups.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to evaluate.
   * @param array $conditions
   *   The conditions configuration.
   *
   * @return string|null
   *   The status to assign, or NULL if no condition matches.
   */
  public function evaluateConditions(ContentEntityInterface $entity, array $conditions): ?string {
    // Sort conditions by weight.
    usort($conditions, fn($a, $b) => ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0));

    foreach ($conditions as $condition) {
      $status = $condition['status'] ?? NULL;
      if (empty($status)) {
        continue;
      }

      // Check if this condition uses the new groups structure.
      if (!empty($condition['groups'])) {
        if ($this->evaluateConditionGroups($entity, $condition)) {
          return $status;
        }
        continue;
      }

      // Legacy: flat plugins array (backwards compatibility).
      $plugins = $condition['plugins'] ?? [];
      if (empty($plugins)) {
        // No plugins configured, condition always matches.
        return $status;
      }

      // All plugins in this condition must match (implicit AND).
      if ($this->evaluatePluginGroup($entity, $plugins)) {
        return $status;
      }
    }

    return NULL;
  }

  /**
   * Evaluates condition groups with AND/OR logic.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to evaluate.
   * @param array $condition
   *   The condition configuration with groups.
   *
   * @return bool
   *   TRUE if the condition matches, FALSE otherwise.
   */
  protected function evaluateConditionGroups(ContentEntityInterface $entity, array $condition): bool {
    $groups = $condition['groups'] ?? [];
    if (empty($groups)) {
      return TRUE;
    }

    $group_operator = strtoupper($condition['group_operator'] ?? 'AND');

    foreach ($groups as $group) {
      $plugins = $group['plugins'] ?? [];
      $group_matches = $this->evaluatePluginGroup($entity, $plugins);

      if ($group_operator === 'OR' && $group_matches) {
        // OR: at least one group must match.
        return TRUE;
      }

      if ($group_operator === 'AND' && !$group_matches) {
        // AND: all groups must match.
        return FALSE;
      }
    }

    // For OR: no group matched.
    // For AND: all groups matched.
    return $group_operator === 'AND';
  }

  /**
   * Evaluates a single group of plugins (all must match - AND logic).
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to evaluate.
   * @param array $plugins
   *   Array of plugin configurations.
   *
   * @return bool
   *   TRUE if all plugins match, FALSE otherwise.
   */
  protected function evaluatePluginGroup(ContentEntityInterface $entity, array $plugins): bool {
    if (empty($plugins)) {
      // Empty group always matches.
      return TRUE;
    }

    foreach ($plugins as $plugin_config) {
      $plugin_id = $plugin_config['plugin_id'] ?? NULL;
      if (!$plugin_id) {
        continue;
      }

      try {
        $plugin = $this->conditionManager->createInstance($plugin_id, $plugin_config['configuration'] ?? []);
        if (!$plugin->evaluate($entity)) {
          return FALSE;
        }
      }
      catch (\Exception $e) {
        // Plugin doesn't exist or error, skip this plugin.
        $this->logger->warning('Error evaluating condition plugin @plugin: @message', [
          '@plugin' => $plugin_id,
          '@message' => $e->getMessage(),
        ]);
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Gets a summary of matched conditions for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param array $conditions
   *   The conditions configuration.
   *
   * @return string
   *   Summary of matched conditions.
   */
  public function getMatchedConditionsSummary(ContentEntityInterface $entity, array $conditions): string {
    $summaries = [];

    foreach ($conditions as $condition) {
      // Check if this condition uses the new groups structure.
      if (!empty($condition['groups'])) {
        foreach ($condition['groups'] as $group) {
          $plugins = $group['plugins'] ?? [];
          foreach ($plugins as $plugin_config) {
            $plugin_id = $plugin_config['plugin_id'] ?? NULL;
            if (!$plugin_id) {
              continue;
            }

            try {
              $plugin = $this->conditionManager->createInstance($plugin_id, $plugin_config['configuration'] ?? []);
              $summaries[] = $plugin->getFormattedValue($entity);
            }
            catch (\Exception $e) {
              // Skip.
            }
          }
        }
      }
      else {
        // Legacy: flat plugins array.
        $plugins = $condition['plugins'] ?? [];
        foreach ($plugins as $plugin_config) {
          $plugin_id = $plugin_config['plugin_id'] ?? NULL;
          if (!$plugin_id) {
            continue;
          }

          try {
            $plugin = $this->conditionManager->createInstance($plugin_id, $plugin_config['configuration'] ?? []);
            $summaries[] = $plugin->getFormattedValue($entity);
          }
          catch (\Exception $e) {
            // Skip.
          }
        }
      }
      // Only show first matching condition.
      break;
    }

    return implode(', ', $summaries);
  }

  /**
   * Updates the lifecycle status of an entity directly in the database.
   *
   * This bypasses the Entity API to avoid changing the 'changed' timestamp.
   * Note: lifecycle fields are non-revisionable, so we only update the data
   * table.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param int|string $entity_id
   *   The entity ID (can be int or string depending on entity type).
   * @param string $status
   *   The lifecycle status to set.
   * @param string|null $condition_details
   *   Optional condition details to store.
   */
  public function updateLifecycleStatus(string $entity_type_id, int|string $entity_id, string $status, ?string $condition_details = NULL): void {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $data_table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();
    $id_key = $entity_type->getKey('id');

    // Build fields to update.
    $fields = ['lifecycle_status' => $status];
    if ($condition_details !== NULL) {
      $fields['lifecycle_condition_details'] = $condition_details;
    }

    // Update data table only (lifecycle fields are non-revisionable).
    $this->database->update($data_table)
      ->fields($fields)
      ->condition($id_key, $entity_id)
      ->execute();

    // Clear entity cache.
    $this->entityTypeManager->getStorage($entity_type_id)->resetCache([$entity_id]);
  }

  /**
   * Gets all enabled bundles with their settings.
   *
   * @return array
   *   Nested array of entity_type => bundle => settings.
   */
  public function getAllEnabledBundles(): array {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    $entity_types = $config->get('entity_types') ?? [];
    $enabled = [];

    foreach ($entity_types as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $settings) {
        if (!empty($settings['enabled'])) {
          $enabled[$entity_type_id][$bundle_id] = $settings;
        }
      }
    }

    return $enabled;
  }

  /**
   * Gets all enabled bundleless entity types with their settings.
   *
   * @return array
   *   Array of entity_type_id => settings.
   */
  public function getAllEnabledBundlelessEntityTypes(): array {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    $bundleless_types = $config->get('bundleless_entity_types') ?? [];
    $enabled = [];

    foreach ($bundleless_types as $entity_type_id => $settings) {
      if (!empty($settings['enabled'])) {
        $enabled[$entity_type_id] = $settings;
      }
    }

    return $enabled;
  }

  /**
   * Finds bundleless entities that need a status update based on conditions.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param array $conditions
   *   Array of condition configurations.
   *
   * @return array
   *   Array of entity data with new status to assign.
   */
  protected function findBundlelessEntitiesForStatusUpdate(
    string $entity_type_id,
    array $conditions,
  ): array {
    $results = [];

    // Load entities via the Entity API to properly evaluate conditions.
    $storage = $this->entityTypeManager->getStorage($entity_type_id);

    // Build query for candidate entities.
    $query = $storage->getQuery()
      ->accessCheck(FALSE);

    // Get entity type definition to check for status key.
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    $status_key = $entity_type->getKey('status');
    if ($status_key) {
      $query->condition($status_key, 1);
    }

    // Exclude entities explicitly excluded from scanning.
    $exclude_group = $query->orConditionGroup()
      ->notExists('lifecycle_exclude')
      ->condition('lifecycle_exclude', 0);
    $query->condition($exclude_group);

    // Allow submodules to alter the query for entity type specific conditions.
    $settings = $this->entityTypeResolver->getBundlelessEntityTypeSettings($entity_type_id);
    $this->moduleHandler->alter('entity_lifecycle_bundleless_query', $query, $entity_type_id, $settings);

    $entity_ids = $query->execute();

    if (empty($entity_ids)) {
      return $results;
    }

    // Load entities in batches.
    foreach (array_chunk($entity_ids, 50) as $batch_ids) {
      $entities = $storage->loadMultiple($batch_ids);

      foreach ($entities as $entity) {
        if (!$entity instanceof ContentEntityInterface) {
          continue;
        }

        $current_status = $entity->hasField('lifecycle_status')
          ? $entity->get('lifecycle_status')->value
          : NULL;

        // Find the first matching condition.
        $new_status = $this->evaluateConditions($entity, $conditions);

        // Skip if no status applies or status hasn't changed.
        if ($new_status === NULL || $new_status === $current_status) {
          continue;
        }

        // Get entity details for the result.
        $label_key = $entity_type->getKey('label');
        $label = $label_key && $entity->hasField($label_key)
          ? $entity->get($label_key)->value
          : "Entity {$entity->id()}";

        $created = $entity->hasField('created') ? (int) $entity->get('created')->value : 0;
        $changed = $entity->hasField('changed') ? (int) $entity->get('changed')->value : 0;

        $result_data = [
          'entity_type' => $entity_type_id,
          'id' => $entity->id(),
          'label' => $label,
          'bundle' => $entity_type_id,
          'created' => $created ? $this->dateFormatter->format($created, 'short') : '-',
          'changed' => $changed ? $this->dateFormatter->format($changed, 'short') : '-',
          'current_status' => $current_status,
          'new_status' => $new_status,
          'matched_conditions' => $this->getMatchedConditionsSummary($entity, $conditions),
        ];

        // Allow submodules to alter scan result data.
        $this->moduleHandler->alter('entity_lifecycle_scan_result', $result_data, $entity, $entity_type_id);

        $results[] = $result_data;
      }
    }

    return $results;
  }

  /**
   * Gets statistics about lifecycle status.
   *
   * @return array
   *   Statistics array.
   */
  public function getStatistics(): array {
    $all_statuses = entity_lifecycle_get_status_options();
    $stats = ['no_status' => 0, 'by_type' => []];

    // Initialize stats for each status.
    foreach (array_keys($all_statuses) as $status_id) {
      $stats[$status_id] = 0;
    }

    // Get statistics for bundled entity types.
    $enabled_bundles = $this->getAllEnabledBundles();

    foreach ($enabled_bundles as $entity_type_id => $bundles) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $data_table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();
      $bundle_key = $entity_type->getKey('bundle');
      $status_key = $entity_type->getKey('status');

      $stats['by_type'][$entity_type_id] = ['no_status' => 0];
      foreach (array_keys($all_statuses) as $status_id) {
        $stats['by_type'][$entity_type_id][$status_id] = 0;
      }

      foreach ($bundles as $bundle_id => $settings) {
        $query = $this->database->select($data_table, 'e');
        $query->addExpression('COUNT(*)', 'count');
        $query->fields('e', ['lifecycle_status']);
        $query->condition("e.{$bundle_key}", $bundle_id);
        $query->groupBy('e.lifecycle_status');

        if ($status_key) {
          $query->condition("e.{$status_key}", 1);
        }

        if ($entity_type->isTranslatable()) {
          $query->condition('e.default_langcode', 1);
        }

        $result = $query->execute();

        foreach ($result as $row) {
          $status = $row->lifecycle_status;
          $count = (int) $row->count;

          if (empty($status) || !isset($all_statuses[$status])) {
            $stats['no_status'] += $count;
            $stats['by_type'][$entity_type_id]['no_status'] += $count;
          }
          else {
            $stats[$status] += $count;
            $stats['by_type'][$entity_type_id][$status] += $count;
          }
        }
      }
    }

    // Get statistics for bundleless entity types.
    $enabled_bundleless = $this->getAllEnabledBundlelessEntityTypes();

    foreach ($enabled_bundleless as $entity_type_id => $settings) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      $data_table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();
      $status_key = $entity_type->getKey('status');

      $stats['by_type'][$entity_type_id] = ['no_status' => 0];
      foreach (array_keys($all_statuses) as $status_id) {
        $stats['by_type'][$entity_type_id][$status_id] = 0;
      }

      $query = $this->database->select($data_table, 'e');
      $query->addExpression('COUNT(*)', 'count');
      $query->fields('e', ['lifecycle_status']);
      $query->groupBy('e.lifecycle_status');

      if ($status_key) {
        $query->condition("e.{$status_key}", 1);
      }

      // Allow submodules to alter the statistics query.
      $this->moduleHandler->alter('entity_lifecycle_stats_query', $query, $entity_type_id);

      $result = $query->execute();

      foreach ($result as $row) {
        $status = $row->lifecycle_status;
        $count = (int) $row->count;

        if (empty($status) || !isset($all_statuses[$status])) {
          $stats['no_status'] += $count;
          $stats['by_type'][$entity_type_id]['no_status'] += $count;
        }
        else {
          $stats[$status] += $count;
          $stats['by_type'][$entity_type_id][$status] += $count;
        }
      }
    }

    return $stats;
  }

}
