<?php

/**
 * @file
 * Hooks provided by the Entity Lifecycle module.
 */

/**
 * @addtogroup hooks
 * @{
 */

/**
 * Registers bundleless entity types for lifecycle tracking.
 *
 * Implement this hook to allow lifecycle tracking for entity types that don't
 * have bundles (like the user entity type). The main Entity Lifecycle module
 * provides the infrastructure, but submodules must register their supported
 * entity types.
 *
 * @return array
 *   An array of entity type IDs that should support lifecycle tracking.
 *   Each entity type must:
 *   - Be a content entity type (not a config entity)
 *   - Not have bundles (or bundle equals entity type ID)
 *   - Have a data table or base table
 *
 * @see \Drupal\entity_lifecycle\Service\EntityTypeResolver::getSupportedBundlelessEntityTypes()
 */
function hook_entity_lifecycle_bundleless_entity_types(): array {
  return ['user'];
}

/**
 * Alters the query for finding bundleless entities to scan.
 *
 * Use this hook to add entity type specific conditions to the query that finds
 * entities for lifecycle status updates. For example, the user submodule uses
 * this to exclude the anonymous user (uid 0).
 *
 * @param \Drupal\Core\Entity\Query\QueryInterface $query
 *   The entity query.
 * @param string $entity_type_id
 *   The entity type ID being queried.
 * @param array $settings
 *   The lifecycle settings for this entity type.
 */
function hook_entity_lifecycle_bundleless_query_alter($query, string $entity_type_id, array $settings): void {
  if ($entity_type_id === 'user') {
    // Exclude anonymous user.
    $query->condition('uid', 0, '>');
  }
}

/**
 * Alters entity data for lifecycle scan results.
 *
 * Use this hook to add or modify entity-specific data in scan results.
 * For example, the user submodule uses this to add the display name and
 * last login time.
 *
 * @param array $data
 *   The entity data array with keys:
 *   - entity_type: The entity type ID.
 *   - id: The entity ID.
 *   - label: The entity label.
 *   - bundle: The bundle (same as entity_type for bundleless).
 *   - created: Formatted creation date.
 *   - changed: Formatted changed date.
 *   - current_status: Current lifecycle status.
 *   - new_status: New lifecycle status to assign.
 *   - matched_conditions: Summary of matched conditions.
 * @param \Drupal\Core\Entity\ContentEntityInterface $entity
 *   The entity being processed.
 * @param string $entity_type_id
 *   The entity type ID.
 */
function hook_entity_lifecycle_scan_result_alter(array &$data, $entity, string $entity_type_id): void {
  if ($entity_type_id === 'user') {
    // Add user-specific data.
    $data['label'] = $entity->getDisplayName();
    $login = (int) $entity->getLastLoginTime();
    $data['last_login'] = $login ? \Drupal::service('date.formatter')->format($login, 'short') : t('Never');
  }
}

/**
 * Alters the database query for lifecycle statistics.
 *
 * Use this hook to add entity type specific conditions to the statistics query.
 *
 * @param \Drupal\Core\Database\Query\SelectInterface $query
 *   The database query.
 * @param string $entity_type_id
 *   The entity type ID being queried.
 */
function hook_entity_lifecycle_stats_query_alter($query, string $entity_type_id): void {
  if ($entity_type_id === 'user') {
    // Exclude anonymous user from statistics.
    $query->condition('e.uid', 0, '>');
  }
}

/**
 * Declares entity types that should not use a specific condition plugin.
 *
 * Implement this hook to exclude certain entity types from using a condition
 * plugin. For example, the user submodule excludes the 'age' condition for
 * user entities because the 'changed' field represents profile updates rather
 * than user activity.
 *
 * @param string $condition_plugin_id
 *   The condition plugin ID (e.g., 'age', 'last_login').
 *
 * @return array
 *   An array of entity type IDs that should NOT use this condition plugin.
 *   Return an empty array if no entity types should be excluded.
 *
 * @see \Drupal\entity_lifecycle\LifecycleConditionBase::isApplicable()
 */
function hook_entity_lifecycle_condition_excluded_entity_types(string $condition_plugin_id): array {
  if ($condition_plugin_id === 'age') {
    // The 'age' condition uses 'changed' field which represents profile updates
    // for users, not user activity. Use last_login condition instead.
    return ['user'];
  }
  return [];
}

/**
 * Alters the entity type options for the lifecycle summary Views area plugin.
 *
 * Use this hook to add your entity type to the dropdown in the Views area
 * plugin configuration. This allows the summary to be used with custom
 * entity types.
 *
 * @param array $options
 *   An associative array of entity type options where keys are entity type IDs
 *   and values are human-readable labels. Default options include 'node' and
 *   'media'.
 *
 * @see \Drupal\entity_lifecycle\Plugin\views\area\LifecycleSummary::buildOptionsForm()
 */
function hook_entity_lifecycle_summary_entity_types_alter(array &$options): void {
  $options['user'] = t('Users');
  $options['my_entity'] = t('My Custom Entity');
}

/**
 * Alters the database query for lifecycle summary statistics.
 *
 * Use this hook to add entity type specific conditions to the summary query.
 * For example, exclude certain entity IDs from the count.
 *
 * @param \Drupal\Core\Database\Query\SelectInterface $query
 *   The database query for counting entities by lifecycle status.
 * @param string $entity_type_id
 *   The entity type ID being queried.
 *
 * @see \Drupal\entity_lifecycle\Plugin\views\area\LifecycleSummary::buildSummary()
 */
function hook_entity_lifecycle_summary_query_alter($query, string $entity_type_id): void {
  if ($entity_type_id === 'user') {
    // Exclude anonymous user from summary statistics.
    $query->condition('e.uid', 0, '>');
  }
}

/**
 * Alters the render information for lifecycle summary output.
 *
 * Use this hook to override the theme, library, and entity label for
 * entity type specific summary rendering. This allows submodules to provide
 * their own templates and styling.
 *
 * @param array $render_info
 *   An associative array with keys:
 *   - theme: The theme hook name (default: 'entity_lifecycle_summary').
 *   - library: The library to attach (default: 'entity_lifecycle/admin').
 *   - entity_label: Human-readable label for the entity type
 *     (default: 'items').
 * @param string $entity_type_id
 *   The entity type ID being rendered.
 *
 * @see \Drupal\entity_lifecycle\Plugin\views\area\LifecycleSummary::buildSummary()
 */
function hook_entity_lifecycle_summary_render_alter(array &$render_info, string $entity_type_id): void {
  if ($entity_type_id === 'user') {
    $render_info['theme'] = 'entity_lifecycle_user_summary';
    $render_info['library'] = 'entity_lifecycle_user/admin';
    $render_info['entity_label'] = t('users');
  }
}

/**
 * Alters the base table to entity type mapping for auto-detection.
 *
 * The lifecycle summary Views area plugin can auto-detect the entity type from
 * the view's base table. Use this hook to add mappings for your entity types.
 *
 * @param array $map
 *   An associative array where keys are base table names and values are
 *   entity type IDs. Default mappings include node and media tables.
 *
 * @see \Drupal\entity_lifecycle\Plugin\views\area\LifecycleSummary::detectEntityType()
 */
function hook_entity_lifecycle_summary_base_tables_alter(array &$map): void {
  $map['users_field_data'] = 'user';
  $map['users'] = 'user';
  $map['my_entity_field_data'] = 'my_entity';
}

/**
 * @} End of "addtogroup hooks".
 */
