<?php

namespace Drupal\entity_lifecycle_entity_usage\Plugin\LifecycleCondition;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Condition based on entity usage count.
 *
 * Evaluates how many times an entity is referenced by other content.
 * Can be used to identify unused, rarely used, or heavily used content.
 *
 * @LifecycleCondition(
 *   id = "entity_usage",
 *   label = @Translation("Entity Usage"),
 *   description = @Translation("Evaluates entity usage count."),
 *   category = "usage",
 *   weight = 25
 * )
 */
class EntityUsageCondition extends LifecycleConditionBase {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'max_usage_count' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    // Check if Entity Usage tracking is enabled for this entity type.
    if (!$this->isTrackingEnabled($entity->getEntityTypeId())) {
      // If tracking is not enabled, we cannot determine usage.
      // Return FALSE to not match the condition.
      return FALSE;
    }

    $max_count = $this->conditionConfiguration['max_usage_count'] ?? 0;
    $usage_count = $this->getValue($entity);

    return $usage_count <= $max_count;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    $entity_type = $entity->getEntityTypeId();
    $entity_id = $entity->id();

    try {
      // Query entity_usage table to count how many times this entity is used.
      $query = $this->database->select('entity_usage', 'eu');
      $query->condition('eu.target_type', $entity_type);
      $query->condition('eu.target_id', $entity_id);
      $query->addExpression('SUM(eu.count)', 'total_count');

      $result = $query->execute()->fetchField();
      return (int) ($result ?? 0);
    }
    catch (\Exception $e) {
      // Table doesn't exist or other database error.
      return 0;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    $count = $this->getValue($entity);
    if ($count === 0) {
      return $this->t('No content references');
    }
    return $this->formatPlural($count, 'Referenced 1 time', 'Referenced @count times');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $config = $this->configFactory->get('entity_usage.settings');
    $enabled_types = $config->get('track_enabled_target_entity_types') ?? [];

    if (!empty($enabled_types)) {
      $form['info'] = [
        '#type' => 'item',
        '#markup' => $this->t('Entity Usage tracking is currently enabled for: @types', [
          '@types' => implode(', ', $enabled_types),
        ]),
      ];
    }

    $form['max_usage_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum usage count'),
      '#description' => $this->t('Content used this many times or fewer matches the condition. Set to 0 to match only completely unused content.<br><strong>Note:</strong> This condition only works for entity types where Entity Usage tracking is enabled.'),
      '#default_value' => $configuration['max_usage_count'] ?? 0,
      '#min' => 0,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $max = $this->conditionConfiguration['max_usage_count'] ?? 0;
    if ($max === 0) {
      return (string) $this->t('not used anywhere');
    }
    return (string) $this->t('usage <= @max', ['@max' => $max]);
  }

  /**
   * Checks if Entity Usage tracking is enabled for the given entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return bool
   *   TRUE if tracking is enabled, FALSE otherwise.
   */
  protected function isTrackingEnabled(string $entity_type_id): bool {
    $config = $this->configFactory->get('entity_usage.settings');
    $enabled_types = $config->get('track_enabled_target_entity_types') ?? [];
    return in_array($entity_type_id, $enabled_types, TRUE);
  }

}
