<?php

namespace Drupal\entity_lifecycle_radioactivity\Plugin\LifecycleCondition;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Condition based on Radioactivity energy level.
 *
 * Low energy indicates content that is not being viewed/accessed frequently,
 * which may indicate it's outdated or less relevant.
 *
 * @LifecycleCondition(
 *   id = "radioactivity",
 *   label = @Translation("Radioactivity (popularity)"),
 *   description = @Translation("Content with low Radioactivity energy (low popularity/engagement)."),
 *   category = "content",
 *   weight = 10
 * )
 */
class RadioactivityCondition extends LifecycleConditionBase {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'max_energy' => 10,
      'field_name' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    $max_energy = $this->conditionConfiguration['max_energy'] ?? 10;
    $energy = $this->getValue($entity);

    if ($energy === NULL) {
      // No radioactivity field found, condition cannot be evaluated.
      return FALSE;
    }

    // Condition is met if energy is below the threshold.
    return $energy <= $max_energy;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    $field_name = $this->getRadioactivityFieldName($entity);
    if (!$field_name) {
      return NULL;
    }

    if (!$entity->hasField($field_name)) {
      return NULL;
    }

    $field = $entity->get($field_name);
    if ($field->isEmpty()) {
      return 0;
    }

    // Handle both radioactivity and radioactivity_reference field types.
    $field_type = $field->getFieldDefinition()->getType();

    if ($field_type === 'radioactivity_reference') {
      // Reference field - get the referenced radioactivity entity.
      /** @var \Drupal\Core\Entity\ContentEntityInterface|null $referenced_entity */
      $referenced_entity = $field->entity;
      if ($referenced_entity && $referenced_entity->hasField('energy')) {
        return (float) $referenced_entity->get('energy')->value;
      }
      return 0;
    }
    elseif ($field_type === 'radioactivity') {
      // Direct radioactivity field.
      return (float) ($field->energy ?? 0);
    }

    return 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    $energy = $this->getValue($entity);
    if ($energy === NULL) {
      return $this->t('N/A');
    }
    return $this->t('@energy energy', ['@energy' => round($energy, 2)]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['max_energy'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum energy'),
      '#description' => $this->t('Content with energy at or below this level is considered low-engagement. Higher energy means more recent/frequent views.'),
      '#default_value' => $configuration['max_energy'] ?? 10,
      '#min' => 0,
      '#step' => 0.1,
    ];

    $form['field_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Field name'),
      '#description' => $this->t('The machine name of the Radioactivity field. Leave empty to auto-detect.'),
      '#default_value' => $configuration['field_name'] ?? '',
      '#placeholder' => 'field_radioactivity',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $max_energy = $this->conditionConfiguration['max_energy'] ?? 10;
    return (string) $this->t('energy <= @max', ['@max' => $max_energy]);
  }

  /**
   * Gets the radioactivity field name for an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The field name or NULL if not found.
   */
  protected function getRadioactivityFieldName(ContentEntityInterface $entity): ?string {
    // Check if a specific field name is configured.
    $configured_field = $this->conditionConfiguration['field_name'] ?? '';
    if (!empty($configured_field)) {
      return $configured_field;
    }

    // Auto-detect radioactivity field.
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    try {
      $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
      foreach ($field_definitions as $field_name => $definition) {
        $type = $definition->getType();
        if ($type === 'radioactivity' || $type === 'radioactivity_reference') {
          return $field_name;
        }
      }
    }
    catch (\Exception $e) {
      // Field definitions not available.
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id): bool {
    // Check if any bundle of this entity type has a radioactivity field.
    try {
      $bundles = $this->entityTypeManager->getStorage($entity_type_id . '_type')->loadMultiple();
      foreach ($bundles as $bundle) {
        $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle->id());
        foreach ($field_definitions as $definition) {
          $type = $definition->getType();
          if ($type === 'radioactivity' || $type === 'radioactivity_reference') {
            return TRUE;
          }
        }
      }
    }
    catch (\Exception $e) {
      // Bundle entity type doesn't exist or other error.
    }

    return FALSE;
  }

}
