<?php

namespace Drupal\entity_lifecycle_user\Plugin\LifecycleCondition;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Drupal\user\UserInterface;

/**
 * Condition based on user account status (active/blocked).
 *
 * Evaluates whether a user account is active or blocked. This allows tracking
 * of blocked accounts that may need review or cleanup.
 *
 * @LifecycleCondition(
 *   id = "account_status",
 *   label = @Translation("Account status"),
 *   description = @Translation("Evaluates based on whether the user account is active or blocked."),
 *   category = "user",
 *   weight = 5,
 *   entity_types = {"user"}
 * )
 */
class AccountStatusCondition extends LifecycleConditionBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'account_status' => 'blocked',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    if (!$entity instanceof UserInterface) {
      return FALSE;
    }

    // Skip anonymous user.
    if ($entity->isAnonymous()) {
      return FALSE;
    }

    $target_status = $this->conditionConfiguration['account_status'] ?? 'blocked';
    $is_active = $entity->isActive();

    if ($target_status === 'active') {
      return $is_active;
    }

    return !$is_active;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    if (!$entity instanceof UserInterface) {
      return NULL;
    }

    return $entity->isActive();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    if (!$entity instanceof UserInterface) {
      return $this->t('N/A');
    }

    return $entity->isActive()
      ? $this->t('Active')
      : $this->t('Blocked');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['account_status'] = [
      '#type' => 'select',
      '#title' => $this->t('Account status'),
      '#description' => $this->t('Select which account status this condition should match.'),
      '#options' => [
        'blocked' => $this->t('Blocked'),
        'active' => $this->t('Active'),
      ],
      '#default_value' => $configuration['account_status'] ?? 'blocked',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $status = $this->conditionConfiguration['account_status'] ?? 'blocked';

    return $status === 'active'
      ? (string) $this->t('active')
      : (string) $this->t('blocked');
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id, ?string $bundle = NULL): bool {
    // This condition only applies to user entities.
    if ($entity_type_id !== 'user') {
      return FALSE;
    }

    return parent::isApplicable($entity_type_id);
  }

}
