<?php

namespace Drupal\entity_lifecycle_user\Plugin\LifecycleCondition;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Condition based on user's last login time.
 *
 * Evaluates whether a user has not logged in for a specified number of months.
 * This is useful for identifying inactive or dormant user accounts.
 *
 * @LifecycleCondition(
 *   id = "last_login",
 *   label = @Translation("Last login"),
 *   description = @Translation("User has not logged in for a specified number of months."),
 *   category = "user",
 *   weight = 0,
 *   entity_types = {"user"}
 * )
 */
class LastLoginCondition extends LifecycleConditionBase {

  use StringTranslationTrait;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->time = $container->get('datetime.time');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'min_months_since_login' => 6,
      'include_never_logged_in' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    if (!$entity instanceof UserInterface) {
      return FALSE;
    }

    // Skip anonymous user.
    if ($entity->isAnonymous()) {
      return FALSE;
    }

    $min_months = $this->conditionConfiguration['min_months_since_login'] ?? 6;
    $include_never = $this->conditionConfiguration['include_never_logged_in'] ?? TRUE;

    $last_login = $entity->getLastLoginTime();

    // Handle users who have never logged in.
    if (empty($last_login) || $last_login === 0) {
      return $include_never;
    }

    // Calculate months since last login.
    $now = $this->time->getRequestTime();
    $months_since_login = ($now - $last_login) / (30 * 24 * 60 * 60);

    return $months_since_login >= $min_months;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    if (!$entity instanceof UserInterface) {
      return NULL;
    }

    $last_login = $entity->getLastLoginTime();

    if (empty($last_login) || $last_login === 0) {
      return NULL;
    }

    // Return months since last login.
    $now = $this->time->getRequestTime();
    return round(($now - $last_login) / (30 * 24 * 60 * 60), 1);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    if (!$entity instanceof UserInterface) {
      return $this->t('N/A');
    }

    $last_login = $entity->getLastLoginTime();

    if (empty($last_login) || $last_login === 0) {
      return $this->t('Never logged in');
    }

    $months = $this->getValue($entity);
    if ($months < 1) {
      $days = round($months * 30);
      return $this->t('Last login @days days ago', ['@days' => $days]);
    }

    return $this->t('Last login @months months ago', ['@months' => round($months)]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['min_months_since_login'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum months since last login'),
      '#description' => $this->t('Users who have not logged in for at least this many months will match this condition.'),
      '#default_value' => $configuration['min_months_since_login'] ?? 6,
      '#min' => 1,
      '#max' => 120,
      '#step' => 1,
    ];

    $form['include_never_logged_in'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include users who have never logged in'),
      '#description' => $this->t('If checked, users who have never logged in will also match this condition.'),
      '#default_value' => $configuration['include_never_logged_in'] ?? TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $min_months = $this->conditionConfiguration['min_months_since_login'] ?? 6;
    $include_never = $this->conditionConfiguration['include_never_logged_in'] ?? TRUE;

    $summary = (string) $this->t('last login >= @months months', ['@months' => $min_months]);

    if ($include_never) {
      $summary .= ' ' . (string) $this->t('(or never)');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id): bool {
    // This condition only applies to user entities.
    return $entity_type_id === 'user';
  }

}
