<?php

namespace Drupal\entity_lifecycle\Drush\Commands;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\entity_lifecycle\Service\LifecycleScanner;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for entity lifecycle management.
 */
class EntityLifecycleCommands extends DrushCommands {

  /**
   * The lifecycle scanner service.
   */
  protected LifecycleScanner $scanner;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  /**
   * Constructs an EntityLifecycleCommands object.
   */
  public function __construct(LifecycleScanner $scanner, ConfigFactoryInterface $config_factory, DateFormatterInterface $date_formatter) {
    parent::__construct();
    $this->scanner = $scanner;
    $this->configFactory = $config_factory;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_lifecycle.scanner'),
      $container->get('config.factory'),
      $container->get('date.formatter')
    );
  }

  /**
   * Scans for outdated entities and marks them for review.
   */
  #[CLI\Command(name: 'entity-lifecycle:scan', aliases: ['lc-scan'])]
  #[CLI\Option(name: 'dry-run', description: 'Only show what would be changed, without making changes.')]
  #[CLI\Option(name: 'entity-type', description: 'Limit to specific entity type (e.g., node, media).')]
  #[CLI\Option(name: 'bundle', description: 'Limit to specific bundle (requires --entity-type).')]
  #[CLI\Usage(name: 'drush entity-lifecycle:scan', description: 'Scan and mark outdated entities.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:scan --dry-run', description: 'Preview what would be marked.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:scan --entity-type=node --bundle=news', description: 'Scan only news content.')]
  public function scan(
    array $options = [
      'dry-run' => FALSE,
      'entity-type' => NULL,
      'bundle' => NULL,
    ],
  ): void {
    $dry_run = $options['dry-run'];
    $entity_type = $options['entity-type'];
    $bundle = $options['bundle'];

    if ($bundle && !$entity_type) {
      $this->io()->error('--bundle requires --entity-type to be specified.');
      return;
    }

    if ($dry_run) {
      $this->io()->note('Running in dry-run mode. No changes will be made.');
    }

    // Show configuration.
    $this->io()->title('Entity Lifecycle Scan');

    $all_enabled = $this->scanner->getAllEnabledBundles();
    $all_bundleless = $this->scanner->getAllEnabledBundlelessEntityTypes();

    if (empty($all_enabled) && empty($all_bundleless)) {
      $this->io()->warning('No entity types/bundles are enabled for scanning.');
      $this->io()->text('Configure at: /admin/config/content/entity-lifecycle');
      return;
    }

    // Filter display based on options.
    $display_bundles = $all_enabled;
    $display_bundleless = $all_bundleless;
    if ($entity_type) {
      $display_bundles = isset($all_enabled[$entity_type])
        ? [$entity_type => $all_enabled[$entity_type]]
        : [];
      $display_bundleless = isset($all_bundleless[$entity_type])
        ? [$entity_type => $all_bundleless[$entity_type]]
        : [];
    }

    if (!empty($display_bundles)) {
      $this->io()->text('Enabled bundles:');
      foreach ($display_bundles as $type_id => $bundles) {
        $this->io()->text("  {$type_id}:");
        foreach ($bundles as $bundle_id => $settings) {
          if ($bundle && $bundle_id !== $bundle) {
            continue;
          }
          $conditions_str = $this->formatConditions($settings['conditions'] ?? []);
          $this->io()->text("    - {$bundle_id}: {$conditions_str}");
        }
      }
    }

    if (!empty($display_bundleless)) {
      $this->io()->text('Enabled bundleless entity types:');
      foreach ($display_bundleless as $type_id => $settings) {
        $conditions_str = $this->formatConditions($settings['conditions'] ?? []);
        $this->io()->text("  - {$type_id}: {$conditions_str}");
      }
    }
    $this->io()->newLine();

    // Run the scan.
    $results = $this->scanner->scanAndMark($dry_run, $entity_type, $bundle);

    if ($results['count'] === 0) {
      $this->io()->success('No outdated entities found matching the criteria.');
      return;
    }

    // Display results table.
    $rows = [];
    foreach ($results['entities'] as $entity) {
      $rows[] = [
        $entity['entity_type'],
        $entity['id'],
        mb_substr($entity['label'], 0, 40) . (mb_strlen($entity['label']) > 40 ? '...' : ''),
        $entity['bundle'],
        $entity['changed'],
        $entity['matched_conditions'] ?? '-',
        $entity['current_status'] ?: '-',
        $entity['new_status'],
      ];
    }

    $this->io()->table(
      ['Type', 'ID', 'Label', 'Bundle', 'Changed', 'Condition', 'From', 'To'],
      $rows
    );

    if ($dry_run) {
      $this->io()->warning(sprintf('%d entity(ies) would be updated.', $results['count']));
    }
    else {
      $this->io()->success(sprintf('%d entity(ies) updated.', $results['count']));
    }
  }

  /**
   * Shows statistics about entity lifecycle status.
   */
  #[CLI\Command(name: 'entity-lifecycle:stats', aliases: ['lc-stats'])]
  #[CLI\Option(name: 'entity-type', description: 'Show stats for specific entity type only.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:stats', description: 'Show lifecycle status statistics.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:stats --entity-type=media', description: 'Show stats for media only.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:stats --entity-type=user', description: 'Show stats for users only.')]
  public function showStats(array $options = ['entity-type' => NULL]): void {
    $entity_type = $options['entity-type'];

    $all_enabled = $this->scanner->getAllEnabledBundles();
    $all_bundleless = $this->scanner->getAllEnabledBundlelessEntityTypes();
    if (empty($all_enabled) && empty($all_bundleless)) {
      $this->io()->warning('No entity types/bundles are enabled for scanning.');
      return;
    }

    $stats = $this->scanner->getStatistics();
    $all_statuses = entity_lifecycle_get_status_options();

    $this->io()->title('Entity Lifecycle Statistics');

    // Overall stats.
    if (!$entity_type) {
      $this->io()->section('Overall');
      $rows = [];
      foreach ($all_statuses as $status_id => $status_label) {
        $rows[] = [$status_label, $stats[$status_id] ?? 0];
      }
      $rows[] = ['No Status', $stats['no_status']];
      $this->io()->table(['Status', 'Count'], $rows);

      $total = array_sum(array_map(fn($id) => $stats[$id] ?? 0, array_keys($all_statuses))) + $stats['no_status'];
      $this->io()->text(sprintf('Total entities: %d', $total));
      $this->io()->newLine();
    }

    // Per entity type stats.
    foreach ($stats['by_type'] as $type_id => $type_stats) {
      if ($entity_type && $type_id !== $entity_type) {
        continue;
      }

      $this->io()->section(ucfirst($type_id));
      $rows = [];
      foreach ($all_statuses as $status_id => $status_label) {
        $rows[] = [$status_label, $type_stats[$status_id] ?? 0];
      }
      $rows[] = ['No Status', $type_stats['no_status']];
      $this->io()->table(['Status', 'Count'], $rows);
    }
  }

  /**
   * Shows the current configuration.
   */
  #[CLI\Command(name: 'entity-lifecycle:config', aliases: ['lc-config'])]
  #[CLI\Usage(name: 'drush entity-lifecycle:config', description: 'Show current lifecycle configuration.')]
  public function showConfig(): void {
    $config = $this->configFactory->get('entity_lifecycle.settings');

    $this->io()->title('Entity Lifecycle Configuration');

    $this->io()->section('Global Settings');
    $this->io()->text(sprintf('Review validity period: %d months', $config->get('review_validity_period') ?? 12));
    $this->io()->text(sprintf('Display banner: %s', $config->get('display_banner') ? 'Yes' : 'No'));

    $banner_roles = $config->get('banner_roles') ?? [];
    if (!empty($banner_roles)) {
      $this->io()->text('Banner roles: ' . implode(', ', $banner_roles));
    }
    $this->io()->newLine();

    $this->io()->section('Entity Types');

    $entity_types = $config->get('entity_types') ?? [];

    $global_validity = $config->get('review_validity_period') ?? 12;

    foreach ($entity_types as $entity_type_id => $bundles) {
      $this->io()->text(ucfirst($entity_type_id) . ':');

      $rows = [];
      foreach ($bundles as $bundle_id => $settings) {
        $validity = !empty($settings['review_validity_months'])
          ? $settings['review_validity_months'] . 'm'
          : $global_validity . 'm (global)';
        $rows[] = [
          $bundle_id,
          !empty($settings['enabled']) ? 'Yes' : 'No',
          !empty($settings['allow_override']) ? 'Yes' : 'No',
          $validity,
          $this->formatConditions($settings['conditions'] ?? []),
        ];
      }

      $this->io()->table(
        ['Bundle', 'Enabled', 'Override', 'Review Validity', 'Conditions'],
        $rows
      );
    }

    $this->io()->text('Configure at: /admin/config/content/entity-lifecycle');
  }

  /**
   * Rebuilds lifecycle information for all entities.
   */
  #[CLI\Command(name: 'entity-lifecycle:rebuild', aliases: ['lc-rebuild'])]
  #[CLI\Option(name: 'dry-run', description: 'Only show what would be changed, without making changes.')]
  #[CLI\Option(name: 'entity-type', description: 'Limit to specific entity type (e.g., node, media).')]
  #[CLI\Option(name: 'bundle', description: 'Limit to specific bundle (requires --entity-type).')]
  #[CLI\Usage(name: 'drush entity-lifecycle:rebuild', description: 'Reset and rebuild all lifecycle information.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:rebuild --dry-run', description: 'Preview what would be rebuilt.')]
  #[CLI\Usage(name: 'drush entity-lifecycle:rebuild --entity-type=node --bundle=news', description: 'Rebuild only news content.')]
  public function rebuild(
    array $options = [
      'dry-run' => FALSE,
      'entity-type' => NULL,
      'bundle' => NULL,
    ],
  ): void {
    $dry_run = $options['dry-run'];
    $entity_type = $options['entity-type'];
    $bundle = $options['bundle'];

    if ($bundle && !$entity_type) {
      $this->io()->error('--bundle requires --entity-type to be specified.');
      return;
    }

    if ($dry_run) {
      $this->io()->note('Running in dry-run mode. No changes will be made.');
    }

    $this->io()->title('Entity Lifecycle Rebuild');

    $all_enabled = $this->scanner->getAllEnabledBundles();
    $all_bundleless = $this->scanner->getAllEnabledBundlelessEntityTypes();

    if (empty($all_enabled) && empty($all_bundleless)) {
      $this->io()->warning('No entity types/bundles are enabled for lifecycle management.');
      $this->io()->text('Configure at: /admin/config/content/entity-lifecycle');
      return;
    }

    // Filter based on options.
    $target_bundles = $all_enabled;
    $target_bundleless = $all_bundleless;

    if ($entity_type) {
      $target_bundles = isset($all_enabled[$entity_type])
        ? [$entity_type => $all_enabled[$entity_type]]
        : [];
      $target_bundleless = isset($all_bundleless[$entity_type])
        ? [$entity_type => $all_bundleless[$entity_type]]
        : [];

      if ($bundle) {
        if (isset($target_bundles[$entity_type][$bundle])) {
          $target_bundles[$entity_type] = [$bundle => $target_bundles[$entity_type][$bundle]];
        }
        else {
          $target_bundles = [];
        }
      }
    }

    if (empty($target_bundles) && empty($target_bundleless)) {
      $this->io()->warning('No matching entity types/bundles found.');
      return;
    }

    $total_rebuilt = 0;

    // Process bundled entity types.
    foreach ($target_bundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $settings) {
        $this->io()->section("Processing {$entity_type_id}:{$bundle_id}");

        $storage = \Drupal::entityTypeManager()->getStorage($entity_type_id);
        $entity_type_def = \Drupal::entityTypeManager()->getDefinition($entity_type_id);
        $bundle_key = $entity_type_def->getKey('bundle');

        // Find all entities with lifecycle fields.
        $query = $storage->getQuery()
          ->accessCheck(FALSE)
          ->condition($bundle_key, $bundle_id);

        $ids = $query->execute();
        $count = count($ids);

        $this->io()->text("Found {$count} entities");

        if (!$dry_run && $count > 0) {
          // Use the shared rebuild method from scanner service.
          $this->io()->text("Resetting and rebuilding lifecycle information...");
          $rebuild_result = $this->scanner->rebuildAndMark(FALSE, $entity_type_id, $bundle_id);

          $results = $rebuild_result['entities'] ?? [];
          $status_counts = [];

          // Count statuses for summary.
          foreach ($results as $result) {
            $status = $result['new_status'] ?? 'unknown';
            $status_counts[$status] = ($status_counts[$status] ?? 0) + 1;
          }

          // Show detailed results table.
          if (!empty($results)) {
            $this->io()->newLine();
            $rows = [];
            foreach ($results as $result) {
              $rows[] = [
                $result['entity_type'],
                $result['id'],
                mb_substr($result['label'], 0, 40) . (mb_strlen($result['label']) > 40 ? '...' : ''),
                $result['bundle'],
                $result['changed'],
                $result['matched_conditions'] ?? '-',
                $result['current_status'] ?: '-',
                $result['new_status'],
              ];
            }

            $this->io()->table(
              ['Type', 'ID', 'Label', 'Bundle', 'Changed', 'Condition', 'From', 'To'],
              $rows
            );
          }

          // Show summary by status.
          $all_statuses = entity_lifecycle_get_status_options();
          $this->io()->newLine();
          $summary_rows = [];
          foreach ($status_counts as $status_id => $count) {
            $status_label = $all_statuses[$status_id] ?? $status_id;
            $summary_rows[] = [$status_label, $count];
          }

          if (!empty($summary_rows)) {
            $this->io()->table(['Status', 'Count'], $summary_rows);
          }

          $total_rebuilt += array_sum($status_counts);
          $this->io()->success("Rebuilt " . array_sum($status_counts) . " entities");
        }
        elseif ($dry_run) {
          $this->io()->text("Would reset and rebuild {$count} entities");
        }
      }
    }

    // Process bundleless entity types.
    foreach ($target_bundleless as $entity_type_id => $settings) {
      $this->io()->section("Processing {$entity_type_id}");

      $storage = \Drupal::entityTypeManager()->getStorage($entity_type_id);
      $entity_type_def = \Drupal::entityTypeManager()->getDefinition($entity_type_id);

      // Find all entities with lifecycle fields.
      $query = $storage->getQuery()->accessCheck(FALSE);

      // Exclude anonymous user for user entity type.
      if ($entity_type_id === 'user') {
        $id_key = $entity_type_def->getKey('id');
        if ($id_key) {
          $query->condition($id_key, 0, '>');
        }
      }

      $ids = $query->execute();
      $count = count($ids);

      $this->io()->text("Found {$count} entities");

      if (!$dry_run && $count > 0) {
        // Step 1: Reset all lifecycle fields to NULL.
        $connection = \Drupal::database();
        $data_table = $entity_type_def->getDataTable() ?: $entity_type_def->getBaseTable();

        $update = $connection->update($data_table)
          ->fields([
            'lifecycle_status' => NULL,
            'lifecycle_last_reviewed' => NULL,
            'lifecycle_condition_details' => NULL,
          ]);

        // Exclude anonymous user for user entity type.
        if ($entity_type_id === 'user') {
          $id_key = $entity_type_def->getKey('id');
          if ($id_key) {
            $update->condition($id_key, 0, '>');
          }
        }

        $update->execute();

        $this->io()->text("Reset lifecycle fields for {$count} entities");

        // Step 2: Evaluate all entities and assign status.
        $default_status = NULL;
        $statuses = \Drupal::entityTypeManager()
          ->getStorage('lifecycle_status')
          ->loadByProperties(['is_default' => TRUE]);
        if (!empty($statuses)) {
          $default_status = key($statuses);
        }

        $status_counts = [];
        $results = [];

        // Load and process entities in batches.
        foreach (array_chunk($ids, 50) as $batch_ids) {
          $entities = $storage->loadMultiple($batch_ids);

          foreach ($entities as $entity) {
            if (!$entity instanceof ContentEntityInterface) {
              continue;
            }

            // Get entity details.
            $changed = $entity->hasField('changed') ? (int) $entity->get('changed')->value : 0;

            // Evaluate conditions for this entity.
            $new_status = $this->scanner->evaluateConditions($entity, $settings['conditions']);

            if ($new_status) {
              // Entity matches a condition.
              $condition_details = $this->scanner->getMatchedConditionsSummary($entity, $settings['conditions']);
              $this->scanner->updateLifecycleStatus($entity_type_id, $entity->id(), $new_status, $condition_details);

              $results[] = [
                'entity_type' => $entity_type_id,
                'id' => $entity->id(),
                'label' => $entity->label(),
                'bundle' => '-',
                'changed' => $changed ? $this->dateFormatter->format($changed, 'short') : '-',
                'matched_conditions' => $condition_details,
                'current_status' => NULL,
                'new_status' => $new_status,
              ];
              $status_counts[$new_status] = ($status_counts[$new_status] ?? 0) + 1;
            }
            elseif ($default_status) {
              // Entity doesn't match - assign default status.
              $this->scanner->updateLifecycleStatus($entity_type_id, $entity->id(), $default_status, 'Content created');

              $results[] = [
                'entity_type' => $entity_type_id,
                'id' => $entity->id(),
                'label' => $entity->label(),
                'bundle' => '-',
                'changed' => $changed ? $this->dateFormatter->format($changed, 'short') : '-',
                'matched_conditions' => 'Content created',
                'current_status' => NULL,
                'new_status' => $default_status,
              ];
              $status_counts[$default_status] = ($status_counts[$default_status] ?? 0) + 1;
            }
          }
        }

        // Show detailed results table.
        if (!empty($results)) {
          $this->io()->newLine();
          $rows = [];
          foreach ($results as $result) {
            $rows[] = [
              $result['entity_type'],
              $result['id'],
              mb_substr($result['label'], 0, 40) . (mb_strlen($result['label']) > 40 ? '...' : ''),
              $result['bundle'],
              $result['changed'],
              $result['matched_conditions'] ?? '-',
              $result['current_status'] ?: '-',
              $result['new_status'],
            ];
          }

          $this->io()->table(
            ['Type', 'ID', 'Label', 'Bundle', 'Changed', 'Condition', 'From', 'To'],
            $rows
          );
        }

        // Show summary by status.
        $all_statuses = entity_lifecycle_get_status_options();
        $this->io()->newLine();
        $summary_rows = [];
        foreach ($status_counts as $status_id => $count) {
          $status_label = $all_statuses[$status_id] ?? $status_id;
          $summary_rows[] = [$status_label, $count];
        }

        if (!empty($summary_rows)) {
          $this->io()->table(['Status', 'Count'], $summary_rows);
        }

        $total_rebuilt += array_sum($status_counts);
        $this->io()->success("Rebuilt " . array_sum($status_counts) . " entities");
      }
      elseif ($dry_run) {
        $this->io()->text("Would reset and rebuild {$count} entities");
      }
    }

    if (!$dry_run) {
      $this->io()->success("Rebuild complete: {$total_rebuilt} entities processed");
    }
    else {
      $this->io()->note('Dry-run complete. Use without --dry-run to apply changes.');
    }
  }

  /**
   * Formats conditions array for display.
   *
   * @param array $conditions
   *   Array of condition definitions.
   *
   * @return string
   *   Formatted string.
   */
  protected function formatConditions(array $conditions): string {
    if (empty($conditions)) {
      return 'no conditions';
    }

    $parts = [];
    foreach ($conditions as $condition) {
      $criteria = [];
      if (!empty($condition['age_months'])) {
        $criteria[] = "age>={$condition['age_months']}m";
      }
      if (!empty($condition['inactive_months'])) {
        $criteria[] = "unchanged>={$condition['inactive_months']}m";
      }
      $criteria_str = implode(' AND ', $criteria) ?: 'always';
      $parts[] = "({$criteria_str}) -> {$condition['status']}";
    }

    return implode('; ', $parts);
  }

}
