<?php

namespace Drupal\entity_lifecycle\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Defines the Lifecycle Status configuration entity.
 *
 * @ConfigEntityType(
 *   id = "lifecycle_status",
 *   label = @Translation("Lifecycle Status"),
 *   label_collection = @Translation("Lifecycle Statuses"),
 *   label_singular = @Translation("lifecycle status"),
 *   label_plural = @Translation("lifecycle statuses"),
 *   label_count = @PluralTranslation(
 *     singular = "@count lifecycle status",
 *     plural = "@count lifecycle statuses",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\entity_lifecycle\LifecycleStatusListBuilder",
 *     "form" = {
 *       "add" = "Drupal\entity_lifecycle\Form\LifecycleStatusForm",
 *       "edit" = "Drupal\entity_lifecycle\Form\LifecycleStatusForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "status",
 *   admin_permission = "administer entity lifecycle",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "weight" = "weight",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "add-form" = "/admin/config/content/entity-lifecycle/statuses/add",
 *     "edit-form" = "/admin/config/content/entity-lifecycle/statuses/{lifecycle_status}/edit",
 *     "delete-form" = "/admin/config/content/entity-lifecycle/statuses/{lifecycle_status}/delete",
 *     "collection" = "/admin/config/content/entity-lifecycle/statuses",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "color",
 *     "weight",
 *     "is_default",
 *     "requires_review",
 *   },
 * )
 */
class LifecycleStatus extends ConfigEntityBase implements ConfigEntityInterface {

  /**
   * The status ID (machine name).
   *
   * @var string
   */
  protected $id;

  /**
   * The status label.
   *
   * @var string
   */
  protected $label;

  /**
   * The status description.
   *
   * @var string
   */
  protected $description = '';

  /**
   * The status color (CSS class suffix).
   *
   * @var string
   */
  protected $color = 'default';

  /**
   * The weight for ordering.
   *
   * @var int
   */
  protected $weight = 0;

  /**
   * Whether this is the default status for new content.
   *
   * @var bool
   */
  protected $is_default = FALSE;

  /**
   * Whether content with this status requires review.
   *
   * @var bool
   */
  protected $requires_review = FALSE;

  /**
   * Gets the status description.
   *
   * @return string
   *   The description.
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * Gets the status color.
   *
   * @return string
   *   The color class suffix.
   */
  public function getColor(): string {
    return $this->color;
  }

  /**
   * Gets the weight.
   *
   * @return int
   *   The weight.
   */
  public function getWeight(): int {
    return $this->weight;
  }

  /**
   * Checks if this is the default status.
   *
   * @return bool
   *   TRUE if default.
   */
  public function isDefault(): bool {
    return (bool) $this->is_default;
  }

  /**
   * Checks if this status requires review.
   *
   * @return bool
   *   TRUE if requires review.
   */
  public function requiresReview(): bool {
    return (bool) $this->requires_review;
  }

}
