<?php

namespace Drupal\entity_lifecycle\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionManager;

/**
 * Builds the lifecycle settings form for bundle configuration.
 *
 * Supports condition groups with AND/OR logic between groups,
 * inspired by the Drupal Views filter groups UI.
 */
class BundleLifecycleFormBuilder {

  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The lifecycle condition manager.
   *
   * @var \Drupal\entity_lifecycle\LifecycleConditionManager
   */
  protected LifecycleConditionManager $conditionManager;

  /**
   * Service IDs for serialization.
   *
   * @var array
   */
  protected array $serviceIds = [];

  /**
   * Constructs a BundleLifecycleFormBuilder object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LifecycleConditionManager $condition_manager,
  ) {
    $this->configFactory = $config_factory;
    $this->conditionManager = $condition_manager;
    $this->serviceIds = [
      'configFactory' => 'config.factory',
      'conditionManager' => 'plugin.manager.lifecycle_condition',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function __sleep(): array {
    $this->serviceIds = [
      'configFactory' => 'config.factory',
      'conditionManager' => 'plugin.manager.lifecycle_condition',
    ];
    return ['serviceIds'];
  }

  /**
   * {@inheritdoc}
   */
  public function __wakeup(): void {
    foreach ($this->serviceIds as $property => $service_id) {
      // @phpcs:ignore DrupalPractice.Objects.GlobalDrupal.GlobalDrupal
      $this->$property = \Drupal::service($service_id);
    }
  }

  /**
   * Builds the lifecycle settings form elements.
   */
  public function buildForm(array &$form, FormStateInterface $form_state, string $entity_type, string $bundle_id): array {
    $config = $this->configFactory->get('entity_lifecycle.settings');

    $settings = $config->get("entity_types.{$entity_type}.{$bundle_id}") ?? [
      'enabled' => FALSE,
      'allow_override' => TRUE,
      'review_validity_months' => NULL,
      'conditions' => [],
    ];

    $form_state->set('lifecycle_entity_type', $entity_type);

    $form['lifecycle'] = [
      '#type' => 'details',
      '#title' => $this->t('Lifecycle settings'),
      '#group' => 'additional_settings',
      '#weight' => 100,
    ];

    $form['lifecycle']['lifecycle_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable lifecycle scanning'),
      '#description' => $this->t('Enable automatic detection of outdated content for this type.'),
      '#default_value' => $settings['enabled'],
    ];

    $form['lifecycle']['lifecycle_allow_override'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow per-entity override'),
      '#description' => $this->t('Allow editors to override lifecycle settings for individual content items.'),
      '#default_value' => $settings['allow_override'],
      '#states' => [
        'visible' => [
          ':input[name="lifecycle_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['lifecycle']['lifecycle_review_validity_months'] = [
      '#type' => 'number',
      '#title' => $this->t('Review validity period'),
      '#description' => $this->t('Number of months a review remains valid before content needs to be reviewed again.'),
      '#default_value' => $settings['review_validity_months'] ?? NULL,
      '#min' => 1,
      '#max' => 120,
      '#field_suffix' => $this->t('months'),
      '#states' => [
        'visible' => [
          ':input[name="lifecycle_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Build the conditions wrapper using the shared method.
    $form['lifecycle']['conditions_wrapper'] = $this->buildConditionsWrapper(
      $form_state,
      $settings['conditions'],
      $entity_type,
      'lifecycle-conditions-wrapper',
      ':input[name="lifecycle_enabled"]'
    );

    return $form;
  }

  /**
   * Builds the conditions wrapper element with condition groups.
   *
   * This method can be used by other forms to build condition groups UI.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $conditions
   *   The existing conditions configuration.
   * @param string $entity_type
   *   The entity type ID.
   * @param string $wrapper_id
   *   The HTML ID for the AJAX wrapper.
   * @param string|null $visible_state
   *   Optional CSS selector for visibility state.
   * @param array $parent_keys
   *   Optional parent keys for form value lookups. For example, if the wrapper
   *   is at $form['entity_lifecycle']['conditions_wrapper'], pass
   *   ['entity_lifecycle', 'conditions_wrapper'] so getValue() lookups work.
   *
   * @return array
   *   The conditions wrapper form element.
   */
  public function buildConditionsWrapper(FormStateInterface $form_state, array $conditions, string $entity_type, string $wrapper_id = 'lifecycle-conditions-wrapper', ?string $visible_state = NULL, array $parent_keys = []): array {
    $status_options = entity_lifecycle_get_status_options();
    $condition_definitions = $this->getConditionDefinitions($entity_type);

    $form_state->set('lifecycle_entity_type', $entity_type);
    $form_state->set('lifecycle_wrapper_id', $wrapper_id);
    $form_state->set('lifecycle_parent_keys', $parent_keys);

    $wrapper = [
      '#type' => 'fieldset',
      '#title' => $this->t('Lifecycle conditions'),
      '#description' => $this->t('Define conditions for automatic status assignment. Conditions are evaluated from top to bottom; the first matching condition wins.'),
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
      '#attached' => [
        'library' => ['entity_lifecycle/admin'],
      ],
    ];

    if ($visible_state) {
      $wrapper['#states'] = [
        'visible' => [
          $visible_state => ['checked' => TRUE],
        ],
      ];
    }

    // Build conditions list.
    $this->buildConditionsListInternal($wrapper, $form_state, $conditions, $condition_definitions, $status_options, $entity_type, $wrapper_id);

    $wrapper['add_condition'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add condition group'),
      '#submit' => [[static::class, 'addConditionSubmit']],
      '#ajax' => [
        'callback' => [static::class, 'conditionsAjaxCallback'],
        'wrapper' => $wrapper_id,
      ],
      '#limit_validation_errors' => [],
    ];

    return $wrapper;
  }

  /**
   * Gets condition plugin definitions for an entity type.
   *
   * This method checks both the annotation-based entity_types restriction
   * and calls the plugin's isApplicable() method for dynamic filtering.
   */
  protected function getConditionDefinitions(string $entity_type): array {
    $definitions = $this->conditionManager->getDefinitions();
    $applicable = [];

    foreach ($definitions as $plugin_id => $definition) {
      // First check annotation-based restriction.
      $entity_types = $definition['entity_types'] ?? [];
      if (!empty($entity_types) && !in_array($entity_type, $entity_types, TRUE)) {
        continue;
      }

      // Then check the plugin's isApplicable() method for dynamic filtering.
      try {
        $plugin = $this->conditionManager->createInstance($plugin_id);
        if (!$plugin->isApplicable($entity_type)) {
          continue;
        }
      }
      catch (\Exception $e) {
        // Plugin instantiation failed, skip it.
        continue;
      }

      $applicable[$plugin_id] = $definition;
    }

    uasort($applicable, fn($a, $b) => ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0));

    return $applicable;
  }

  /**
   * Builds the condition groups list (internal method).
   *
   * @param array $wrapper
   *   The wrapper element to add conditions to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $conditions
   *   The conditions configuration.
   * @param array $condition_definitions
   *   Available condition plugin definitions.
   * @param array $status_options
   *   Status options for the select.
   * @param string $entity_type
   *   The entity type ID.
   * @param string $wrapper_id
   *   The AJAX wrapper ID.
   */
  protected function buildConditionsListInternal(array &$wrapper, FormStateInterface $form_state, array $conditions, array $condition_definitions, array $status_options, string $entity_type, string $wrapper_id): void {
    $conditions = $this->normalizeConditions($conditions);

    $num_conditions = $form_state->get('num_conditions');
    if ($num_conditions === NULL) {
      $num_conditions = count($conditions) > 0 ? count($conditions) : 0;
      $form_state->set('num_conditions', $num_conditions);
    }

    $condition_type_options = [];
    foreach ($condition_definitions as $plugin_id => $definition) {
      $condition_type_options[$plugin_id] = (string) $definition['label'];
    }

    // Use a table structure for proper tabledrag support.
    $wrapper['conditions'] = [
      '#type' => 'table',
      '#tree' => TRUE,
      '#header' => [],
      '#empty' => $this->t('No condition groups defined yet.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'condition-group-weight',
        ],
      ],
      '#attributes' => ['class' => ['lifecycle-conditions-list']],
    ];

    // Check which group should stay open (after AJAX actions).
    $active_group = $form_state->get('active_condition_group');

    for ($i = 0; $i < $num_conditions; $i++) {
      $condition = $conditions[$i] ?? [];
      $current_status = $condition['status'] ?? '';
      $status_label = $current_status ? ($status_options[$current_status] ?? $current_status) : $this->t('No status selected');

      // Table row with draggable class.
      $wrapper['conditions'][$i]['#attributes']['class'][] = 'draggable';
      $wrapper['conditions'][$i]['#weight'] = $condition['weight'] ?? $i;

      // Determine if this details should be open.
      $is_open = empty($current_status) || ($active_group === $i);

      // Single cell containing the collapsible details.
      $wrapper['conditions'][$i]['group'] = [
        '#type' => 'details',
        '#title' => $this->t('Assign: @status', ['@status' => $status_label]),
        '#open' => $is_open,
        '#attributes' => ['class' => ['lifecycle-condition-group']],
      ];

      // Status selector inside the details.
      $wrapper['conditions'][$i]['group']['status'] = [
        '#type' => 'select',
        '#title' => $this->t('Assign status'),
        '#options' => ['' => $this->t('- Select -')] + $status_options,
        '#default_value' => $current_status,
        '#ajax' => [
          'callback' => [static::class, 'conditionsAjaxCallback'],
          'wrapper' => $wrapper_id,
        ],
      ];

      // Conditions container (plugins).
      $wrapper['conditions'][$i]['group']['filters'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['lifecycle-conditions-container']],
      ];

      // Build individual conditions (plugins).
      $this->buildConditionsForGroupInternal($wrapper, $form_state, $i, $condition, $condition_definitions, $condition_type_options, $wrapper_id);

      // Remove button inside the details.
      $wrapper['conditions'][$i]['group']['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove group'),
        '#name' => 'remove_condition_' . $i,
        '#submit' => [[static::class, 'removeConditionSubmit']],
        '#ajax' => [
          'callback' => [static::class, 'conditionsAjaxCallback'],
          'wrapper' => $wrapper_id,
        ],
        '#limit_validation_errors' => [],
        '#attributes' => ['class' => ['button--small', 'button--danger']],
      ];

      // Hidden weight field for tabledrag.
      $wrapper['conditions'][$i]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight'),
        '#title_display' => 'invisible',
        '#default_value' => $condition['weight'] ?? $i,
        '#delta' => 50,
        '#attributes' => ['class' => ['condition-group-weight']],
      ];
    }
  }

  /**
   * Builds conditions (plugins) for a condition group (internal method).
   *
   * @param array $wrapper
   *   The wrapper element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param int $condition_index
   *   The condition group index.
   * @param array $condition
   *   The condition configuration.
   * @param array $condition_definitions
   *   Available condition plugin definitions.
   * @param array $condition_type_options
   *   Condition type options for select.
   * @param string $wrapper_id
   *   The AJAX wrapper ID.
   */
  protected function buildConditionsForGroupInternal(array &$wrapper, FormStateInterface $form_state, int $condition_index, array $condition, array $condition_definitions, array $condition_type_options, string $wrapper_id): void {
    $plugins = $condition['plugins'] ?? [];

    $num_plugins_key = 'num_plugins_' . $condition_index;
    $num_plugins = $form_state->get($num_plugins_key);
    if ($num_plugins === NULL) {
      $num_plugins = count($plugins) > 0 ? count($plugins) : 1;
      $form_state->set($num_plugins_key, $num_plugins);
    }

    $wrapper['conditions'][$condition_index]['group']['filters']['plugins'] = [
      '#type' => 'table',
      '#header' => [],
      '#empty' => $this->t('No conditions added. Add a condition to this group.'),
      '#attributes' => ['class' => ['lifecycle-plugins-table']],
    ];

    // Get parent keys for getValue lookups.
    $parent_keys = $form_state->get('lifecycle_parent_keys') ?? [];

    for ($p = 0; $p < $num_plugins; $p++) {
      $plugin_config = $plugins[$p] ?? [];

      // Build the getValue path including parent keys if set.
      $value_path = array_merge($parent_keys, [
        'conditions', $condition_index, 'group', 'filters', 'plugins', $p, 'plugin', 'plugin_type',
      ]);
      $selected_type = $form_state->getValue($value_path);

      // Try getUserInput as fallback for AJAX rebuilds when getValue
      // returns NULL.
      if ($selected_type === NULL) {
        $user_input = $form_state->getUserInput();
        $selected_type = $this->getNestedValue($user_input, $value_path);
      }

      if ($selected_type === NULL) {
        $selected_type = $plugin_config['plugin_id'] ?? NULL;
      }

      if ($selected_type === NULL && !empty($condition_type_options)) {
        $selected_type = array_key_first($condition_type_options);
      }

      // Plugin type and config in one cell.
      $wrapper['conditions'][$condition_index]['group']['filters']['plugins'][$p]['plugin'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['lifecycle-plugin-row']],
      ];

      // Show AND divider between conditions (not before first).
      if ($p > 0) {
        $wrapper['conditions'][$condition_index]['group']['filters']['plugins'][$p]['plugin']['and_divider'] = [
          '#type' => 'markup',
          '#markup' => '<div class="lifecycle-and-divider"><span class="lifecycle-and-indicator">' . $this->t('AND') . '</span></div>',
          '#weight' => -100,
        ];
      }

      $wrapper['conditions'][$condition_index]['group']['filters']['plugins'][$p]['plugin']['plugin_type'] = [
        '#type' => 'select',
        '#title' => $this->t('Condition type'),
        '#title_display' => 'invisible',
        '#options' => $condition_type_options,
        '#default_value' => $selected_type,
        '#ajax' => [
          'callback' => [static::class, 'conditionsAjaxCallback'],
          'wrapper' => $wrapper_id,
        ],
      ];

      // Plugin configuration inline.
      if ($selected_type && isset($condition_definitions[$selected_type])) {
        $saved_config = [];
        if (($plugin_config['plugin_id'] ?? '') === $selected_type) {
          $saved_config = $plugin_config['configuration'] ?? [];
        }

        try {
          $plugin = $this->conditionManager->createInstance($selected_type, $saved_config);
          $plugin_form = $plugin->buildConfigurationForm([], $saved_config);

          foreach ($plugin_form as $key => $element) {
            $wrapper['conditions'][$condition_index]['group']['filters']['plugins'][$p]['plugin']['configuration'][$key] = $element;
          }
        }
        catch (\Exception $e) {
          // Plugin instantiation failed.
        }
      }

      // Remove link.
      $wrapper['conditions'][$condition_index]['group']['filters']['plugins'][$p]['operations'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_plugin_' . $condition_index . '_' . $p,
        '#submit' => [[static::class, 'removePluginSubmit']],
        '#ajax' => [
          'callback' => [static::class, 'conditionsAjaxCallback'],
          'wrapper' => $wrapper_id,
        ],
        '#limit_validation_errors' => [],
        '#attributes' => ['class' => ['link', 'action-link', 'action-link--danger']],
      ];
    }

    // Add condition button.
    $wrapper['conditions'][$condition_index]['group']['filters']['add_plugin'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add condition'),
      '#name' => 'add_plugin_' . $condition_index,
      '#submit' => [[static::class, 'addPluginSubmit']],
      '#ajax' => [
        'callback' => [static::class, 'conditionsAjaxCallback'],
        'wrapper' => $wrapper_id,
      ],
      '#limit_validation_errors' => [],
      '#attributes' => ['class' => ['button--small']],
    ];
  }

  /**
   * Normalizes legacy conditions to flat plugins structure.
   *
   * Can be used by other modules to normalize conditions before processing.
   *
   * @param array $conditions
   *   The conditions array.
   *
   * @return array
   *   Normalized conditions with flat plugins structure.
   */
  public function normalizeConditions(array $conditions): array {
    foreach ($conditions as &$condition) {
      // Already has plugins directly - new format.
      if (!empty($condition['plugins']) && !isset($condition['groups'])) {
        continue;
      }

      // Legacy format with groups - flatten to plugins.
      if (!empty($condition['groups'])) {
        $plugins = [];
        foreach ($condition['groups'] as $group) {
          if (!empty($group['plugins'])) {
            $plugins = array_merge($plugins, $group['plugins']);
          }
        }
        $condition['plugins'] = $plugins;
        unset($condition['groups'], $condition['group_operator']);
      }
    }

    return $conditions;
  }

  /**
   * Saves the lifecycle settings from the form.
   */
  public function saveSettings(FormStateInterface $form_state, string $entity_type, string $bundle_id): void {
    // Skip saving during AJAX rebuilds.
    if (static::isAjaxRebuild($form_state)) {
      return;
    }

    $config = $this->configFactory->getEditable('entity_lifecycle.settings');

    // Use the shared method to extract conditions.
    $conditions = static::extractConditionsFromFormState($form_state);

    $review_validity = $form_state->getValue('lifecycle_review_validity_months');

    $settings = [
      'enabled' => (bool) $form_state->getValue('lifecycle_enabled'),
      'allow_override' => (bool) $form_state->getValue('lifecycle_allow_override'),
      'review_validity_months' => !empty($review_validity) ? (int) $review_validity : NULL,
      'conditions' => $conditions,
    ];

    $config->set("entity_types.{$entity_type}.{$bundle_id}", $settings)->save();
  }

  /**
   * Extracts conditions array from form state values.
   *
   * This method can be used by any form that uses the condition groups UI.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The extracted conditions array ready for saving.
   */
  public static function extractConditionsFromFormState(FormStateInterface $form_state): array {
    $conditions = [];
    $conditions_values = $form_state->getValue('conditions');

    if (!is_array($conditions_values)) {
      return $conditions;
    }

    foreach ($conditions_values as $condition_index => $condition_row) {
      // Get status from group sub-element.
      $group = $condition_row['group'] ?? [];
      $status = $group['status'] ?? '';

      if (empty($status)) {
        continue;
      }

      $plugins = [];
      $filters = $group['filters'] ?? [];
      $plugins_values = $filters['plugins'] ?? [];

      foreach ($plugins_values as $plugin_row) {
        $plugin_data = $plugin_row['plugin'] ?? [];
        $plugin_type = $plugin_data['plugin_type'] ?? NULL;
        if (!$plugin_type) {
          continue;
        }

        $plugin_configuration = $plugin_data['configuration'] ?? [];

        // Filter out empty configuration values, but keep the plugin even if
        // it has no configuration (e.g., catch-all plugin).
        $filtered_config = [];
        foreach ($plugin_configuration as $key => $config_value) {
          if (!empty($config_value) || $config_value === '0' || $config_value === 0) {
            $filtered_config[$key] = $config_value;
          }
        }

        $plugins[] = [
          'plugin_id' => $plugin_type,
          'configuration' => $filtered_config,
        ];
      }

      if (!empty($plugins)) {
        $conditions[] = [
          'status' => $status,
          'weight' => (int) ($condition_row['weight'] ?? $condition_index),
          'plugins' => $plugins,
        ];
      }
    }

    usort($conditions, fn($a, $b) => $a['weight'] <=> $b['weight']);

    return $conditions;
  }

  /**
   * Checks if this is an AJAX rebuild that should skip saving.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return bool
   *   TRUE if this is an AJAX rebuild, FALSE otherwise.
   */
  public static function isAjaxRebuild(FormStateInterface $form_state): bool {
    $triggering_element = $form_state->getTriggeringElement();
    if (!$triggering_element) {
      return FALSE;
    }

    $button_name = $triggering_element['#name'] ?? '';
    if (str_starts_with($button_name, 'remove_') ||
        str_starts_with($button_name, 'add_') ||
        $button_name === 'op') {
      $button_value = $triggering_element['#value'] ?? '';
      if (is_object($button_value)) {
        $button_value = $button_value->getUntranslatedString();
      }
      if (in_array($button_value, [
        'Add condition group',
        'Add condition',
        'Remove',
        'Remove group',
      ])) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Ajax callback.
   *
   * Finds the conditions wrapper element dynamically based on form structure.
   */
  public static function conditionsAjaxCallback(array &$form, FormStateInterface $form_state) {
    // Try different possible locations for the conditions wrapper.
    // For node/media forms (via buildForm).
    if (isset($form['lifecycle']['conditions_wrapper'])) {
      return $form['lifecycle']['conditions_wrapper'];
    }
    // For user forms (via hook_form_alter with entity_lifecycle container).
    if (isset($form['entity_lifecycle']['conditions_wrapper'])) {
      return $form['entity_lifecycle']['conditions_wrapper'];
    }
    // Direct conditions_wrapper (unlikely but possible).
    if (isset($form['conditions_wrapper'])) {
      return $form['conditions_wrapper'];
    }

    // Fallback: return the whole form if wrapper not found.
    return $form;
  }

  /**
   * Add condition group submit handler.
   */
  public static function addConditionSubmit(array &$form, FormStateInterface $form_state): void {
    $num_conditions = $form_state->get('num_conditions');
    $form_state->set('num_conditions', $num_conditions + 1);
    $form_state->set('num_plugins_' . $num_conditions, 1);
    $form_state->setRebuild();
  }

  /**
   * Remove condition group submit handler.
   */
  public static function removeConditionSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'] ?? '';

    if (preg_match('/remove_condition_(\d+)/', $button_name, $matches)) {
      $condition_index = (int) $matches[1];
      $num_conditions = $form_state->get('num_conditions');

      // Shift plugins counts for all conditions after the removed one.
      for ($i = $condition_index; $i < $num_conditions - 1; $i++) {
        $next_plugins = $form_state->get('num_plugins_' . ($i + 1));
        $form_state->set('num_plugins_' . $i, $next_plugins);
      }

      $last = $num_conditions - 1;
      $form_state->set('num_plugins_' . $last, NULL);
      $form_state->set('num_conditions', $num_conditions - 1);
    }

    $form_state->setRebuild();
  }

  /**
   * Add condition (plugin) submit handler.
   */
  public static function addPluginSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'] ?? '';

    if (preg_match('/add_plugin_(\d+)/', $button_name, $matches)) {
      $condition_index = (int) $matches[1];
      $num_plugins_key = 'num_plugins_' . $condition_index;
      $num_plugins = $form_state->get($num_plugins_key) ?? 1;
      $form_state->set($num_plugins_key, $num_plugins + 1);
      // Keep this group open after rebuild.
      $form_state->set('active_condition_group', $condition_index);
    }

    $form_state->setRebuild();
  }

  /**
   * Remove condition (plugin) submit handler.
   */
  public static function removePluginSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'] ?? '';

    if (preg_match('/remove_plugin_(\d+)_(\d+)/', $button_name, $matches)) {
      $condition_index = (int) $matches[1];

      $num_plugins_key = 'num_plugins_' . $condition_index;
      $num_plugins = $form_state->get($num_plugins_key) ?? 1;

      if ($num_plugins > 0) {
        $form_state->set($num_plugins_key, $num_plugins - 1);
      }
      // Keep this group open after rebuild.
      $form_state->set('active_condition_group', $condition_index);
    }

    $form_state->setRebuild();
  }

  /**
   * Gets a nested value from an array using a key path.
   *
   * @param array $array
   *   The array to search.
   * @param array $keys
   *   The key path as an array.
   *
   * @return mixed|null
   *   The value at the path, or NULL if not found.
   */
  protected function getNestedValue(array $array, array $keys): mixed {
    foreach ($keys as $key) {
      if (!is_array($array) || !array_key_exists($key, $array)) {
        return NULL;
      }
      $array = $array[$key];
    }
    return $array;
  }

}
