<?php

namespace Drupal\entity_lifecycle\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form handler for Lifecycle Status add/edit forms.
 */
class LifecycleStatusForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\entity_lifecycle\Entity\LifecycleStatus $status */
    $status = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $status->label(),
      '#description' => $this->t('The human-readable name of this status.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $status->id(),
      '#machine_name' => [
        'exists' => '\Drupal\entity_lifecycle\Entity\LifecycleStatus::load',
      ],
      '#disabled' => !$status->isNew(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $status->getDescription(),
      '#description' => $this->t('A brief description of this status.'),
      '#rows' => 2,
    ];

    $form['color'] = [
      '#type' => 'select',
      '#title' => $this->t('Color'),
      '#options' => [
        'default' => $this->t('Default (gray)'),
        'info' => $this->t('Info (blue)'),
        'success' => $this->t('Success (green)'),
        'warning' => $this->t('Warning (yellow)'),
        'danger' => $this->t('Danger (red)'),
      ],
      '#default_value' => $status->getColor(),
      '#description' => $this->t('The color used for displaying this status in banners and lists.'),
    ];

    $form['is_default'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Default status'),
      '#default_value' => $status->isDefault(),
      '#description' => $this->t('If checked, new content will be assigned this status by default. Only one status can be the default.'),
    ];

    $form['requires_review'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Requires review'),
      '#default_value' => $status->requiresReview(),
      '#description' => $this->t('If checked, content with this status will show a review banner and appear in the "Needs Review" list.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // If this is set as default, unset other defaults.
    if ($form_state->getValue('is_default')) {
      $storage = $this->entityTypeManager->getStorage('lifecycle_status');
      $statuses = $storage->loadMultiple();
      foreach ($statuses as $existing_status) {
        if ($existing_status->id() !== $this->entity->id() && $existing_status->isDefault()) {
          // We'll handle this in save().
          break;
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    /** @var \Drupal\entity_lifecycle\Entity\LifecycleStatus $status */
    $status = $this->entity;

    // If this is set as default, unset other defaults.
    if ($status->isDefault()) {
      $storage = $this->entityTypeManager->getStorage('lifecycle_status');
      $statuses = $storage->loadMultiple();
      foreach ($statuses as $existing_status) {
        if ($existing_status->id() !== $status->id() && $existing_status->isDefault()) {
          $existing_status->set('is_default', FALSE);
          $existing_status->save();
        }
      }
    }

    $result = $status->save();

    $message_args = ['%label' => $status->label()];
    if ($result === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('Created new lifecycle status %label.', $message_args));
    }
    else {
      $this->messenger()->addStatus($this->t('Saved lifecycle status %label.', $message_args));
    }

    $form_state->setRedirectUrl($status->toUrl('collection'));
    return $result;
  }

}
