<?php

namespace Drupal\entity_lifecycle\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\entity_lifecycle\Service\LifecycleScanner;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for scanning a specific bundle.
 */
class ScanBundleConfirmForm extends ConfirmFormBase {

  /**
   * The lifecycle scanner service.
   */
  protected LifecycleScanner $scanner;

  /**
   * The entity type ID.
   */
  protected string $entityTypeId;

  /**
   * The bundle ID.
   */
  protected string $bundle;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->scanner = $container->get('entity_lifecycle.scanner');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_lifecycle_scan_bundle_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to scan %type (%bundle)?', [
      '%type' => $this->entityTypeId,
      '%bundle' => $this->bundle,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This will evaluate lifecycle conditions for all entities of this type and update their lifecycle status if conditions match.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity_lifecycle.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $entity_type_id = NULL, ?string $bundle = NULL): array {
    $this->entityTypeId = $entity_type_id;
    $this->bundle = $bundle;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Set up batch operation.
    $batch = [
      'title' => $this->t('Scanning @type (@bundle)...', [
        '@type' => $this->entityTypeId,
        '@bundle' => $this->bundle,
      ]),
      'operations' => [
        [
          [EntityLifecycleSettingsForm::class, 'batchScanOperation'],
          [$this->entityTypeId, $this->bundle],
        ],
      ],
      'finished' => [EntityLifecycleSettingsForm::class, 'batchScanFinished'],
    ];

    batch_set($batch);

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
