<?php

namespace Drupal\entity_lifecycle;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Lifecycle Condition plugins.
 */
abstract class LifecycleConditionBase extends PluginBase implements LifecycleConditionInterface {

  /**
   * The condition configuration.
   *
   * @var array
   */
  protected array $conditionConfiguration = [];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->setConfiguration($configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function description(): string {
    return (string) ($this->pluginDefinition['description'] ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->conditionConfiguration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->conditionConfiguration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id): bool {
    // Check plugin annotation restriction.
    $entity_types = $this->pluginDefinition['entity_types'] ?? [];
    if (!empty($entity_types) && !in_array($entity_type_id, $entity_types, TRUE)) {
      return FALSE;
    }

    // Check hook-based exclusions from submodules.
    $excluded_types = $this->getExcludedEntityTypes();
    if (in_array($entity_type_id, $excluded_types, TRUE)) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Gets entity types excluded from this condition via hooks.
   *
   * @return array
   *   Array of entity type IDs that should not use this condition.
   */
  protected function getExcludedEntityTypes(): array {
    $excluded = [];
    $plugin_id = $this->getPluginId();

    // Invoke hook_entity_lifecycle_condition_excluded_entity_types().
    \Drupal::moduleHandler()->invokeAllWith(
      'entity_lifecycle_condition_excluded_entity_types',
      function (callable $hook, string $module) use (&$excluded, $plugin_id) {
        $module_excluded = $hook($plugin_id);
        if (is_array($module_excluded)) {
          $excluded = array_merge($excluded, $module_excluded);
        }
      }
    );

    return array_unique($excluded);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    return (string) $this->getValue($entity);
  }

}
