<?php

namespace Drupal\entity_lifecycle;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

/**
 * Interface for Lifecycle Condition plugins.
 *
 * Lifecycle conditions evaluate whether content meets certain criteria
 * that indicate it may be outdated or needs review.
 */
interface LifecycleConditionInterface extends PluginInspectionInterface, ConfigurableInterface, ContainerFactoryPluginInterface {

  /**
   * Returns the label of the condition.
   *
   * @return string
   *   The condition label.
   */
  public function label(): string;

  /**
   * Returns the description of the condition.
   *
   * @return string
   *   The condition description.
   */
  public function description(): string;

  /**
   * Evaluates the condition for a given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to evaluate.
   *
   * @return bool
   *   TRUE if the condition is met (entity matches the criteria).
   */
  public function evaluate(ContentEntityInterface $entity): bool;

  /**
   * Returns the current value for the condition.
   *
   * This is used for display purposes in reports and scans.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to get the value for.
   *
   * @return mixed
   *   The current value (e.g., age in months, energy level, link count).
   */
  public function getValue(ContentEntityInterface $entity): mixed;

  /**
   * Returns the formatted value for display.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to get the value for.
   *
   * @return string
   *   The formatted value for display.
   */
  public function getFormattedValue(ContentEntityInterface $entity): string;

  /**
   * Returns the configuration form elements for this condition.
   *
   * @param array $form
   *   The form array.
   * @param array $configuration
   *   The current configuration values.
   *
   * @return array
   *   Form elements for configuring this condition.
   */
  public function buildConfigurationForm(array $form, array $configuration): array;

  /**
   * Returns a summary of the condition configuration.
   *
   * @return string
   *   A human-readable summary (e.g., "age >= 24 months").
   */
  public function summary(): string;

  /**
   * Checks if this condition is applicable for the given entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return bool
   *   TRUE if applicable.
   */
  public function isApplicable(string $entity_type_id): bool;

}
