<?php

namespace Drupal\entity_lifecycle;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Plugin manager for Lifecycle Condition plugins.
 */
class LifecycleConditionManager extends DefaultPluginManager {

  /**
   * Constructs a LifecycleConditionManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct(
      'Plugin/LifecycleCondition',
      $namespaces,
      $module_handler,
      'Drupal\entity_lifecycle\LifecycleConditionInterface',
      'Drupal\entity_lifecycle\Annotation\LifecycleCondition'
    );
    $this->alterInfo('lifecycle_condition_info');
    $this->setCacheBackend($cache_backend, 'lifecycle_condition_plugins');
  }

  /**
   * Gets all condition plugins applicable for an entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return \Drupal\entity_lifecycle\LifecycleConditionInterface[]
   *   Array of condition plugins.
   */
  public function getApplicableConditions(string $entity_type_id): array {
    $conditions = [];
    $definitions = $this->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $entity_types = $definition['entity_types'] ?? [];
      if (empty($entity_types) || in_array($entity_type_id, $entity_types, TRUE)) {
        $conditions[$plugin_id] = $this->createInstance($plugin_id);
      }
    }

    // Sort by weight.
    uasort($conditions, function ($a, $b) {
      $weight_a = $a->getPluginDefinition()['weight'] ?? 0;
      $weight_b = $b->getPluginDefinition()['weight'] ?? 0;
      return $weight_a <=> $weight_b;
    });

    return $conditions;
  }

  /**
   * Gets options for a select list of conditions.
   *
   * @param string|null $entity_type_id
   *   Optional entity type ID to filter by.
   *
   * @return array
   *   Array of condition labels keyed by plugin ID.
   */
  public function getConditionOptions(?string $entity_type_id = NULL): array {
    $options = [];
    $definitions = $this->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $entity_types = $definition['entity_types'] ?? [];
      if ($entity_type_id === NULL || empty($entity_types) || in_array($entity_type_id, $entity_types, TRUE)) {
        $options[$plugin_id] = $definition['label'];
      }
    }

    return $options;
  }

}
