<?php

namespace Drupal\entity_lifecycle\Plugin\LifecycleCondition;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;

/**
 * Condition based on entity published status.
 *
 * Evaluates whether content is published or unpublished. This allows tracking
 * of unpublished content that may need review or cleanup.
 *
 * @LifecycleCondition(
 *   id = "published_status",
 *   label = @Translation("Published status"),
 *   description = @Translation("Evaluates based on whether content is published or unpublished."),
 *   category = "content",
 *   weight = 5
 * )
 */
class PublishedStatusCondition extends LifecycleConditionBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'published_status' => 'unpublished',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    if (!$entity instanceof EntityPublishedInterface) {
      // Entity doesn't support published status, condition doesn't apply.
      return FALSE;
    }

    $target_status = $this->conditionConfiguration['published_status'] ?? 'unpublished';
    $is_published = $entity->isPublished();

    if ($target_status === 'published') {
      return $is_published;
    }

    return !$is_published;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    if (!$entity instanceof EntityPublishedInterface) {
      return NULL;
    }

    return $entity->isPublished();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    if (!$entity instanceof EntityPublishedInterface) {
      return $this->t('N/A');
    }

    return $entity->isPublished()
      ? $this->t('Published')
      : $this->t('Unpublished');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['published_status'] = [
      '#type' => 'select',
      '#title' => $this->t('Content status'),
      '#description' => $this->t('Select which content status this condition should match.'),
      '#options' => [
        'unpublished' => $this->t('Unpublished'),
        'published' => $this->t('Published'),
      ],
      '#default_value' => $configuration['published_status'] ?? 'unpublished',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $status = $this->conditionConfiguration['published_status'] ?? 'unpublished';

    return $status === 'published'
      ? (string) $this->t('published')
      : (string) $this->t('unpublished');
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id, ?string $bundle = NULL): bool {
    // This condition only applies to entities that implement
    // EntityPublishedInterface (nodes, media, etc.) but NOT users.
    // Users have their own account status condition.
    if ($entity_type_id === 'user') {
      return FALSE;
    }

    return parent::isApplicable($entity_type_id);
  }

}
