<?php

namespace Drupal\entity_lifecycle\Plugin\views\filter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\InOperator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter for the lifecycle status of an entity.
 *
 * @ingroup views_filter_handlers
 */
#[ViewsFilter("lifecycle_status_filter")]
class LifecycleStatusFilter extends InOperator {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;

    // Set default operator to 'in' if not set.
    if (empty($this->operator)) {
      $this->operator = 'in';
    }
  }

  /**
   * {@inheritdoc}
   */
  public function init($view, $display, &$options = NULL) {
    parent::init($view, $display, $options);

    // Ensure operator is valid.
    if (empty($this->operator)) {
      $this->operator = 'in';
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getValueOptions() {
    if (!empty($this->valueOptions)) {
      return $this->valueOptions;
    }

    $this->valueOptions = [];

    try {
      $statuses = $this->entityTypeManager
        ->getStorage('lifecycle_status')
        ->loadMultiple();

      // Sort by weight.
      uasort($statuses, function ($a, $b) {
        return $a->getWeight() <=> $b->getWeight();
      });

      foreach ($statuses as $status) {
        $this->valueOptions[$status->id()] = $status->label();
      }
    }
    catch (\Exception $e) {
      // If entity type doesn't exist, return empty options.
    }

    return $this->valueOptions;
  }

  /**
   * {@inheritdoc}
   */
  public function acceptExposedInput($input) {
    // Ensure the identifier key exists in the input to prevent PHP warnings.
    if (!empty($this->options['expose']['identifier'])) {
      $identifier = $this->options['expose']['identifier'];
      if (!isset($input[$identifier])) {
        $input[$identifier] = [];
      }
    }

    return parent::acceptExposedInput($input);
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state) {
    parent::valueForm($form, $form_state);

    // Ensure the exposed filter shows a proper select dropdown.
    if ($form_state->get('exposed')) {
      $identifier = $this->options['expose']['identifier'];
      if (isset($form[$identifier])) {
        $form[$identifier]['#type'] = 'select';
        $form[$identifier]['#multiple'] = !empty($this->options['expose']['multiple']);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function operators() {
    $operators = [
      'in' => [
        'title' => $this->t('Is one of'),
        'short' => $this->t('in'),
        'short_single' => $this->t('='),
        'method' => 'opSimple',
        'values' => 1,
      ],
      'not in' => [
        'title' => $this->t('Is not one of'),
        'short' => $this->t('not in'),
        'short_single' => $this->t('!='),
        'method' => 'opSimple',
        'values' => 1,
      ],
      // Include '=' and '!=' for backwards compatibility with views that may
      // have been saved using the default list_string filter before our custom
      // filter was applied.
      '=' => [
        'title' => $this->t('Is equal to'),
        'short' => $this->t('='),
        'method' => 'opSimple',
        'values' => 1,
      ],
      '!=' => [
        'title' => $this->t('Is not equal to'),
        'short' => $this->t('!='),
        'method' => 'opSimple',
        'values' => 1,
      ],
      'empty' => [
        'title' => $this->t('Is empty (NULL)'),
        'method' => 'opEmpty',
        'short' => $this->t('empty'),
        'values' => 0,
      ],
      'not empty' => [
        'title' => $this->t('Is not empty (NOT NULL)'),
        'method' => 'opEmpty',
        'short' => $this->t('not empty'),
        'values' => 0,
      ],
    ];

    return $operators;
  }

  /**
   * {@inheritdoc}
   */
  public function validate() {
    $errors = parent::validate();

    // Ensure operator is valid for this filter.
    if (!empty($this->operator)) {
      $operators = $this->operators();
      if (!isset($operators[$this->operator])) {
        $errors[] = $this->t('The operator @operator is invalid for filter: @filter', [
          '@operator' => $this->operator,
          '@filter' => $this->adminLabel(),
        ]);
      }
    }

    return $errors;
  }

}
