<?php

namespace Drupal\entity_lifecycle\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Service for resolving which entity types support lifecycle tracking.
 *
 * This service provides a central place to determine which entity types
 * are enabled for lifecycle tracking, replacing hardcoded node/media lists.
 *
 * Supports two categories of entity types:
 * - Bundled entity types (node, media): Configured per-bundle via bundle forms.
 * - Bundleless entity types (user): Configured at entity type level.
 */
class EntityTypeResolver {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info service.
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The module handler.
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The last installed schema repository.
   */
  protected EntityLastInstalledSchemaRepositoryInterface $schemaRepository;

  /**
   * Static cache for supported entity types.
   *
   * @var array|null
   */
  protected ?array $supportedEntityTypes = NULL;

  /**
   * Static cache for supported bundleless entity types.
   *
   * @var array|null
   */
  protected ?array $supportedBundlelessEntityTypes = NULL;

  /**
   * Static cache for enabled entity types.
   *
   * @var array|null
   */
  protected ?array $enabledEntityTypes = NULL;

  /**
   * Constructs an EntityTypeResolver object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityLastInstalledSchemaRepositoryInterface $schema_repository
   *   The last installed schema repository.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityTypeBundleInfoInterface $bundle_info,
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
    EntityFieldManagerInterface $entity_field_manager,
    EntityLastInstalledSchemaRepositoryInterface $schema_repository,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->bundleInfo = $bundle_info;
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->entityFieldManager = $entity_field_manager;
    $this->schemaRepository = $schema_repository;
  }

  /**
   * Gets all content entity types that can potentially support lifecycle.
   *
   * These are entity types that:
   * - Are content entities (not config entities)
   * - Have bundles (so we can configure per-bundle settings)
   * - Have a 'changed' field (to track last modification)
   *
   * @return \Drupal\Core\Entity\ContentEntityTypeInterface[]
   *   Array of entity type definitions keyed by entity type ID.
   */
  public function getSupportedEntityTypes(): array {
    if ($this->supportedEntityTypes !== NULL) {
      return $this->supportedEntityTypes;
    }

    $this->supportedEntityTypes = [];
    $definitions = $this->entityTypeManager->getDefinitions();

    foreach ($definitions as $entity_type_id => $entity_type) {
      // Skip non-content entities.
      if (!$entity_type instanceof ContentEntityTypeInterface) {
        continue;
      }

      // Must have a bundle key (supports bundles).
      $bundle_key = $entity_type->getKey('bundle');
      if (empty($bundle_key)) {
        continue;
      }

      // Must have a data table or base table for our fields.
      if (!$entity_type->getDataTable() && !$entity_type->getBaseTable()) {
        continue;
      }

      // Check if entity type has a bundle entity type (node_type, media_type).
      // This ensures we can hook into bundle configuration forms.
      $bundle_entity_type = $entity_type->getBundleEntityType();
      if (empty($bundle_entity_type)) {
        continue;
      }

      $this->supportedEntityTypes[$entity_type_id] = $entity_type;
    }

    // Sort by label.
    uasort($this->supportedEntityTypes, function ($a, $b) {
      return strnatcasecmp((string) $a->getLabel(), (string) $b->getLabel());
    });

    return $this->supportedEntityTypes;
  }

  /**
   * Gets all bundleless content entity types that can support lifecycle.
   *
   * These are entity types that:
   * - Are content entities (not config entities)
   * - Do NOT have bundles (or bundle equals entity type ID)
   * - Have a data/base table
   * - Are registered via hook_entity_lifecycle_bundleless_entity_types()
   *
   * @return \Drupal\Core\Entity\ContentEntityTypeInterface[]
   *   Array of entity type definitions keyed by entity type ID.
   */
  public function getSupportedBundlelessEntityTypes(): array {
    if ($this->supportedBundlelessEntityTypes !== NULL) {
      return $this->supportedBundlelessEntityTypes;
    }

    $this->supportedBundlelessEntityTypes = [];
    $definitions = $this->entityTypeManager->getDefinitions();

    // Get list of bundleless entity types registered by submodules.
    $allowed_bundleless_types = $this->getRegisteredBundlelessEntityTypes();

    foreach ($definitions as $entity_type_id => $entity_type) {
      // Skip non-content entities.
      if (!$entity_type instanceof ContentEntityTypeInterface) {
        continue;
      }

      // Only allow registered bundleless types.
      if (!in_array($entity_type_id, $allowed_bundleless_types, TRUE)) {
        continue;
      }

      // Must have a data table or base table for our fields.
      if (!$entity_type->getDataTable() && !$entity_type->getBaseTable()) {
        continue;
      }

      // Must NOT have a bundle entity type (bundleless).
      $bundle_entity_type = $entity_type->getBundleEntityType();
      if (!empty($bundle_entity_type)) {
        continue;
      }

      $this->supportedBundlelessEntityTypes[$entity_type_id] = $entity_type;
    }

    // Sort by label.
    uasort($this->supportedBundlelessEntityTypes, function ($a, $b) {
      return strnatcasecmp((string) $a->getLabel(), (string) $b->getLabel());
    });

    return $this->supportedBundlelessEntityTypes;
  }

  /**
   * Gets bundleless entity types registered by submodules.
   *
   * @return string[]
   *   Array of entity type IDs.
   */
  protected function getRegisteredBundlelessEntityTypes(): array {
    $types = [];

    // Invoke hook_entity_lifecycle_bundleless_entity_types().
    $this->moduleHandler->invokeAllWith(
      'entity_lifecycle_bundleless_entity_types',
      function (callable $hook, string $module) use (&$types) {
        $module_types = $hook();
        if (is_array($module_types)) {
          $types = array_merge($types, $module_types);
        }
      }
    );

    return array_unique($types);
  }

  /**
   * Checks if an entity type is bundleless.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return bool
   *   TRUE if the entity type is bundleless.
   */
  public function isBundlelessEntityType(string $entity_type_id): bool {
    return isset($this->getSupportedBundlelessEntityTypes()[$entity_type_id]);
  }

  /**
   * Gets entity types that are enabled for lifecycle tracking.
   *
   * @return string[]
   *   Array of enabled entity type IDs.
   */
  public function getEnabledEntityTypes(): array {
    if ($this->enabledEntityTypes !== NULL) {
      return $this->enabledEntityTypes;
    }

    $config = $this->configFactory->get('entity_lifecycle.settings');
    $this->enabledEntityTypes = $config->get('enabled_entity_types') ?? ['node', 'media'];

    // Filter to only include currently supported types (bundled).
    $supported = array_keys($this->getSupportedEntityTypes());
    $this->enabledEntityTypes = array_values(
      array_intersect($this->enabledEntityTypes, $supported)
    );

    // Also include enabled bundleless entity types.
    $enabled_bundleless = $config->get('enabled_bundleless_entity_types') ?? [];
    $supported_bundleless = array_keys($this->getSupportedBundlelessEntityTypes());
    $enabled_bundleless = array_values(
      array_intersect($enabled_bundleless, $supported_bundleless)
    );

    $this->enabledEntityTypes = array_unique(
      array_merge($this->enabledEntityTypes, $enabled_bundleless)
    );

    return $this->enabledEntityTypes;
  }

  /**
   * Gets enabled bundleless entity types.
   *
   * @return string[]
   *   Array of enabled bundleless entity type IDs.
   */
  public function getEnabledBundlelessEntityTypes(): array {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    $enabled = $config->get('enabled_bundleless_entity_types') ?? [];
    $supported = array_keys($this->getSupportedBundlelessEntityTypes());
    return array_values(array_intersect($enabled, $supported));
  }

  /**
   * Checks if an entity type is enabled for lifecycle tracking.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return bool
   *   TRUE if the entity type is enabled.
   */
  public function isEntityTypeEnabled(string $entity_type_id): bool {
    return in_array($entity_type_id, $this->getEnabledEntityTypes(), TRUE);
  }

  /**
   * Gets bundles for an entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   Array of bundles with their info.
   */
  public function getBundlesForEntityType(string $entity_type_id): array {
    return $this->bundleInfo->getBundleInfo($entity_type_id);
  }

  /**
   * Gets the bundle entity type ID for a content entity type.
   *
   * @param string $entity_type_id
   *   The content entity type ID (e.g., 'node').
   *
   * @return string|null
   *   The bundle entity type ID (e.g., 'node_type'), or NULL if not found.
   */
  public function getBundleEntityTypeId(string $entity_type_id): ?string {
    $supported = $this->getSupportedEntityTypes();
    if (isset($supported[$entity_type_id])) {
      return $supported[$entity_type_id]->getBundleEntityType();
    }
    return NULL;
  }

  /**
   * Gets the content entity type ID from a bundle entity type ID.
   *
   * @param string $bundle_entity_type_id
   *   The bundle entity type ID (e.g., 'node_type').
   *
   * @return string|null
   *   The content entity type ID (e.g., 'node'), or NULL if not found.
   */
  public function getEntityTypeFromBundleEntityType(string $bundle_entity_type_id): ?string {
    $supported = $this->getSupportedEntityTypes();
    foreach ($supported as $entity_type_id => $entity_type) {
      if ($entity_type->getBundleEntityType() === $bundle_entity_type_id) {
        return $entity_type_id;
      }
    }
    return NULL;
  }

  /**
   * Gets the form ID pattern for bundle entity forms.
   *
   * @param string $entity_type_id
   *   The content entity type ID.
   *
   * @return string|null
   *   The form ID for the bundle edit form, or NULL if not determinable.
   */
  public function getBundleFormId(string $entity_type_id): ?string {
    $bundle_entity_type = $this->getBundleEntityTypeId($entity_type_id);
    if ($bundle_entity_type) {
      // Standard Drupal pattern: {bundle_entity_type}_form.
      return $bundle_entity_type . '_form';
    }
    return NULL;
  }

  /**
   * Clears the internal cache.
   *
   * Call this when configuration changes.
   */
  public function resetCache(): void {
    $this->supportedEntityTypes = NULL;
    $this->supportedBundlelessEntityTypes = NULL;
    $this->enabledEntityTypes = NULL;
  }

  /**
   * Gets the settings for a bundleless entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   The settings array or empty array if not configured.
   */
  public function getBundlelessEntityTypeSettings(string $entity_type_id): array {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    return $config->get("bundleless_entity_types.{$entity_type_id}") ?? [];
  }

  /**
   * Installs lifecycle base fields for a specific entity type.
   *
   * This method can be used by the main module and submodules to install
   * base fields when lifecycle tracking is enabled for an entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID (e.g., 'node', 'user').
   * @param string $provider
   *   The module providing the fields (e.g., 'entity_lifecycle',
   *   'entity_lifecycle_user').
   *
   * @return int
   *   The number of fields installed.
   */
  public function installBaseFieldsForEntityType(string $entity_type_id, string $provider): int {
    $definition_update_manager = \Drupal::entityDefinitionUpdateManager();
    $installed_count = 0;

    // Check if entity type exists.
    try {
      $this->entityTypeManager->getDefinition($entity_type_id);
    }
    catch (\Exception $e) {
      \Drupal::logger('entity_lifecycle')->warning('Entity type @type not found, skipping field installation.', [
        '@type' => $entity_type_id,
      ]);
      return 0;
    }

    // Clear field definition cache to pick up new fields.
    $this->entityFieldManager->clearCachedFieldDefinitions();

    // Disable caches to get fresh field definitions including our new fields.
    $this->entityFieldManager->useCaches(FALSE);
    $storage_definitions = $this->entityFieldManager->getFieldStorageDefinitions($entity_type_id);
    $this->entityFieldManager->useCaches(TRUE);

    // Get already installed field definitions.
    $installed_storage_definitions = $this->schemaRepository->getLastInstalledFieldStorageDefinitions($entity_type_id);

    // Find fields that are defined but not yet installed.
    foreach (array_diff_key($storage_definitions, $installed_storage_definitions) as $storage_definition) {
      /** @var \Drupal\Core\Field\FieldStorageDefinitionInterface $storage_definition */
      // Only install fields provided by the specified provider.
      if ($storage_definition->getProvider() === $provider) {
        try {
          $definition_update_manager->installFieldStorageDefinition(
            $storage_definition->getName(),
            $entity_type_id,
            $provider,
            $storage_definition
          );
          \Drupal::logger('entity_lifecycle')->notice('Installed field @field for @type', [
            '@field' => $storage_definition->getName(),
            '@type' => $entity_type_id,
          ]);
          $installed_count++;
        }
        catch (\Exception $e) {
          \Drupal::logger('entity_lifecycle')->error('Failed to install field @field for @type: @message', [
            '@field' => $storage_definition->getName(),
            '@type' => $entity_type_id,
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    return $installed_count;
  }

}
