<?php

namespace Drupal\entity_lifecycle_linkchecker\Plugin\LifecycleCondition;

use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Condition based on the number of broken links in content.
 *
 * Content with broken links may be outdated or need maintenance.
 *
 * @LifecycleCondition(
 *   id = "broken_links",
 *   label = @Translation("Broken links"),
 *   description = @Translation("Content containing broken links (HTTP errors)."),
 *   category = "content",
 *   weight = 20
 * )
 */
class BrokenLinksCondition extends LifecycleConditionBase {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->database = $container->get('database');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'min_broken_links' => 1,
      'include_status_codes' => '404,500,502,503',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id): bool {
    // Broken links condition is not applicable to user entities.
    // Users typically don't have URL fields that linkchecker tracks.
    return $entity_type_id !== 'user';
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    $min_broken = $this->conditionConfiguration['min_broken_links'] ?? 1;
    $broken_count = $this->getValue($entity);

    return $broken_count >= $min_broken;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    $status_codes = $this->getStatusCodes();
    if (empty($status_codes)) {
      return 0;
    }

    $entity_type = $entity->getEntityTypeId();
    $entity_id = $entity->id();

    try {
      // Query linkchecker_link table for broken links belonging to this entity.
      $query = $this->database->select('linkchecker_link', 'll');
      $query->condition('ll.entity_id__target_id', $entity_id);
      $query->condition('ll.entity_id__target_type', $entity_type);
      $query->condition('ll.code', $status_codes, 'IN');
      $query->condition('ll.status', 1);

      return (int) $query->countQuery()->execute()->fetchField();
    }
    catch (\Exception $e) {
      // Table doesn't exist or other database error.
      return 0;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    $count = $this->getValue($entity);
    if ($count === 0) {
      return $this->t('No broken links');
    }
    return $this->formatPlural($count, '1 broken link', '@count broken links');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['min_broken_links'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum broken links'),
      '#description' => $this->t('Content with at least this many broken links matches the condition.'),
      '#default_value' => $configuration['min_broken_links'] ?? 1,
      '#min' => 1,
    ];

    $form['include_status_codes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('HTTP status codes'),
      '#description' => $this->t('Comma-separated list of HTTP status codes to consider as broken (e.g., 404,500,502,503).'),
      '#default_value' => $configuration['include_status_codes'] ?? '404,500,502,503',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $min = $this->conditionConfiguration['min_broken_links'] ?? 1;
    return (string) $this->t('broken links >= @min', ['@min' => $min]);
  }

  /**
   * Gets the status codes to check for.
   *
   * @return array
   *   Array of status codes as integers.
   */
  protected function getStatusCodes(): array {
    $codes_string = $this->conditionConfiguration['include_status_codes'] ?? '404,500,502,503';
    $codes = array_map('trim', explode(',', $codes_string));
    return array_filter(array_map('intval', $codes));
  }

}
