<?php

namespace Drupal\entity_lifecycle_user\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\entity_lifecycle\Form\BundleLifecycleFormBuilder;
use Drupal\entity_lifecycle\Service\EntityTypeResolver;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for user lifecycle settings.
 *
 * Uses the shared BundleLifecycleFormBuilder for condition groups UI.
 */
class UserLifecycleSettingsForm extends ConfigFormBase {

  /**
   * The bundle lifecycle form builder.
   *
   * @var \Drupal\entity_lifecycle\Form\BundleLifecycleFormBuilder
   */
  protected BundleLifecycleFormBuilder $formBuilder;

  /**
   * The entity type resolver.
   *
   * @var \Drupal\entity_lifecycle\Service\EntityTypeResolver
   */
  protected EntityTypeResolver $entityTypeResolver;

  /**
   * Constructs a UserLifecycleSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\entity_lifecycle\Form\BundleLifecycleFormBuilder $form_builder
   *   The bundle lifecycle form builder.
   * @param \Drupal\entity_lifecycle\Service\EntityTypeResolver $entity_type_resolver
   *   The entity type resolver.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    BundleLifecycleFormBuilder $form_builder,
    EntityTypeResolver $entity_type_resolver,
  ) {
    parent::__construct($config_factory);
    $this->formBuilder = $form_builder;
    $this->entityTypeResolver = $entity_type_resolver;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_lifecycle.bundle_form_builder'),
      $container->get('entity_lifecycle.entity_type_resolver'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_lifecycle_user_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['entity_lifecycle.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('entity_lifecycle.settings');
    $user_settings = $config->get('bundleless_entity_types.user') ?? [];

    // Normalize legacy conditions format.
    $conditions = $this->normalizeConditions($user_settings['conditions'] ?? []);

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable lifecycle tracking for users'),
      '#description' => $this->t('When enabled, user accounts can be scanned for inactivity based on configured conditions.'),
      '#default_value' => $user_settings['enabled'] ?? FALSE,
    ];

    $form['allow_override'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow per-user override'),
      '#description' => $this->t('Allow administrators to exclude specific users or set custom review periods.'),
      '#default_value' => $user_settings['allow_override'] ?? TRUE,
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['review_validity_months'] = [
      '#type' => 'number',
      '#title' => $this->t('Default review validity (months)'),
      '#description' => $this->t('Default period in months before a user account needs review.'),
      '#default_value' => $user_settings['review_validity_months'] ?? 12,
      '#min' => 1,
      '#max' => 120,
      '#field_suffix' => $this->t('months'),
      '#states' => [
        'visible' => [
          ':input[name="enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Use the shared form builder for conditions UI.
    $form['conditions_wrapper'] = $this->formBuilder->buildConditionsWrapper(
      $form_state,
      $conditions,
      'user',
      'user-lifecycle-conditions-wrapper',
      ':input[name="enabled"]'
    );

    return parent::buildForm($form, $form_state);
  }

  /**
   * Normalizes legacy conditions to new flat plugins structure.
   *
   * Converts old format with nested groups to flat plugins format.
   *
   * @param array $conditions
   *   The conditions array.
   *
   * @return array
   *   Normalized conditions with flat plugins structure.
   */
  protected function normalizeConditions(array $conditions): array {
    foreach ($conditions as &$condition) {
      // Already has flat plugins - new format.
      if (!empty($condition['plugins']) && !isset($condition['groups'])) {
        continue;
      }

      // Legacy format with nested groups - flatten to plugins.
      if (!empty($condition['groups'])) {
        $plugins = [];
        foreach ($condition['groups'] as $group) {
          if (!empty($group['plugins'])) {
            $plugins = array_merge($plugins, $group['plugins']);
          }
        }
        $condition['plugins'] = $plugins;
        unset($condition['groups'], $condition['group_operator']);
      }
    }

    return $conditions;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Skip saving during AJAX rebuilds.
    if (BundleLifecycleFormBuilder::isAjaxRebuild($form_state)) {
      return;
    }

    $config = $this->configFactory->getEditable('entity_lifecycle.settings');

    // Use the shared method to extract conditions.
    $conditions = BundleLifecycleFormBuilder::extractConditionsFromFormState($form_state);

    $review_validity = $form_state->getValue('review_validity_months');

    $user_settings = [
      'enabled' => (bool) $form_state->getValue('enabled'),
      'allow_override' => (bool) $form_state->getValue('allow_override'),
      'review_validity_months' => !empty($review_validity) ? (int) $review_validity : NULL,
      'conditions' => $conditions,
    ];

    // Get existing bundleless settings.
    $bundleless_types = $config->get('bundleless_entity_types') ?? [];
    $bundleless_types['user'] = $user_settings;
    $config->set('bundleless_entity_types', $bundleless_types);

    // Update enabled list.
    $enabled_bundleless = $config->get('enabled_bundleless_entity_types') ?? [];
    if ($user_settings['enabled']) {
      if (!in_array('user', $enabled_bundleless, TRUE)) {
        $enabled_bundleless[] = 'user';
      }
    }
    else {
      $enabled_bundleless = array_values(array_diff($enabled_bundleless, ['user']));
    }
    $config->set('enabled_bundleless_entity_types', $enabled_bundleless);

    $config->save();

    // Reset entity type resolver cache.
    $this->entityTypeResolver->resetCache();

    parent::submitForm($form, $form_state);
  }

}
