<?php

namespace Drupal\entity_lifecycle_user\Plugin\LifecycleCondition;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\entity_lifecycle\LifecycleConditionBase;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Condition based on user account age.
 *
 * Evaluates whether a user account is older than a specified number of months.
 * This is useful in combination with other conditions to identify old accounts
 * that are also inactive.
 *
 * @LifecycleCondition(
 *   id = "account_age",
 *   label = @Translation("Account age"),
 *   description = @Translation("User account is older than a specified number of months."),
 *   category = "user",
 *   weight = 10,
 *   entity_types = {"user"}
 * )
 */
class AccountAgeCondition extends LifecycleConditionBase {

  use StringTranslationTrait;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->time = $container->get('datetime.time');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'min_account_age_months' => 12,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    if (!$entity instanceof UserInterface) {
      return FALSE;
    }

    // Skip anonymous user.
    if ($entity->isAnonymous()) {
      return FALSE;
    }

    $min_months = $this->conditionConfiguration['min_account_age_months'] ?? 12;
    $created = $entity->getCreatedTime();

    if (empty($created)) {
      return FALSE;
    }

    // Calculate account age in months.
    $now = $this->time->getRequestTime();
    $account_age_months = ($now - $created) / (30 * 24 * 60 * 60);

    return $account_age_months >= $min_months;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    if (!$entity instanceof UserInterface) {
      return NULL;
    }

    $created = $entity->getCreatedTime();

    if (empty($created)) {
      return NULL;
    }

    // Return account age in months.
    $now = $this->time->getRequestTime();
    return round(($now - $created) / (30 * 24 * 60 * 60), 1);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormattedValue(ContentEntityInterface $entity): string {
    if (!$entity instanceof UserInterface) {
      return $this->t('N/A');
    }

    $months = $this->getValue($entity);

    if ($months === NULL) {
      return $this->t('Unknown');
    }

    if ($months < 1) {
      return $this->t('Less than 1 month old');
    }

    if ($months < 12) {
      return $this->t('@months months old', ['@months' => round($months)]);
    }

    $years = floor($months / 12);
    $remaining_months = round($months % 12);

    if ($remaining_months > 0) {
      return $this->t('@years years, @months months old', [
        '@years' => $years,
        '@months' => $remaining_months,
      ]);
    }

    return $this->t('@years years old', ['@years' => $years]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['min_account_age_months'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum account age (months)'),
      '#description' => $this->t('User accounts older than this many months will match this condition. Use this in combination with activity conditions to identify old inactive accounts.'),
      '#default_value' => $configuration['min_account_age_months'] ?? 12,
      '#min' => 1,
      '#max' => 240,
      '#step' => 1,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $min_months = $this->conditionConfiguration['min_account_age_months'] ?? 12;
    return (string) $this->t('account age >= @months months', ['@months' => $min_months]);
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(string $entity_type_id): bool {
    // This condition only applies to user entities.
    return $entity_type_id === 'user';
  }

}
