<?php

namespace Drupal\entity_lifecycle\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Service for checking lifecycle status of entities.
 */
class LifecycleStatusChecker {

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The current user.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a LifecycleStatusChecker object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Checks if an entity needs a review banner displayed.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if a banner should be displayed.
   */
  public function needsReviewBanner(ContentEntityInterface $entity): bool {
    if (!$entity->hasField('lifecycle_status')) {
      return FALSE;
    }

    $status = $entity->get('lifecycle_status')->value;
    if (empty($status)) {
      return FALSE;
    }

    // Check if this status requires review using config entity.
    if (!$this->statusRequiresReview($status)) {
      return FALSE;
    }

    // Check if user has permission to see banner.
    if (!$this->userCanSeeBanner()) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Checks if a status requires review.
   *
   * @param string $status_id
   *   The status ID.
   *
   * @return bool
   *   TRUE if the status requires review.
   */
  protected function statusRequiresReview(string $status_id): bool {
    try {
      $status = $this->entityTypeManager
        ->getStorage('lifecycle_status')
        ->load($status_id);

      if ($status) {
        return $status->requiresReview();
      }
    }
    catch (\Exception $e) {
      // If entity type doesn't exist, return FALSE.
    }

    return FALSE;
  }

  /**
   * Checks if the current user can see lifecycle banners.
   *
   * @return bool
   *   TRUE if the user can see banners.
   */
  protected function userCanSeeBanner(): bool {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    $banner_roles = $config->get('banner_roles') ?? [];

    // If no roles specified, show to all authenticated users with permission.
    if (empty($banner_roles)) {
      return $this->currentUser->hasPermission('view entity lifecycle dashboard');
    }

    // Check if user has one of the specified roles.
    $user_roles = $this->currentUser->getRoles();
    return !empty(array_intersect($banner_roles, $user_roles));
  }

  /**
   * Gets the lifecycle status of an entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return string|null
   *   The status string or NULL if not set.
   */
  public function getStatus(ContentEntityInterface $entity): ?string {
    if (!$entity->hasField('lifecycle_status')) {
      return NULL;
    }

    return $entity->get('lifecycle_status')->value;
  }

  /**
   * Checks if an entity is overdue for review.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if the entity is overdue.
   */
  public function isOverdueForReview(ContentEntityInterface $entity): bool {
    $status = $this->getStatus($entity);
    if (empty($status)) {
      return FALSE;
    }

    return $this->statusRequiresReview($status);
  }

  /**
   * Gets the review validity period for an entity in months.
   *
   * Uses bundle-specific setting if configured, otherwise falls back to global.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return int
   *   The review validity period in months.
   */
  public function getReviewValidityMonths(ContentEntityInterface $entity): int {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    $entity_type = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    // Check for bundle-specific setting.
    $bundle_settings = $config->get("entity_types.{$entity_type}.{$bundle}");
    if (!empty($bundle_settings['review_validity_months'])) {
      return (int) $bundle_settings['review_validity_months'];
    }

    // Fall back to global setting.
    return (int) ($config->get('review_validity_period') ?? 12);
  }

  /**
   * Checks if lifecycle is enabled for an entity's bundle.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return bool
   *   TRUE if lifecycle is enabled for this bundle.
   */
  public function isEnabledForBundle(ContentEntityInterface $entity): bool {
    $config = $this->configFactory->get('entity_lifecycle.settings');
    $entity_type = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    $bundle_settings = $config->get("entity_types.{$entity_type}.{$bundle}");
    return !empty($bundle_settings['enabled']);
  }

}
