# Entity Lifecycle - Developer Documentation

This document covers the architecture, APIs, and extension points of the
Entity Lifecycle module.

## Architecture Overview

### Services

| Service | Class | Description |
|---------|-------|-------------|
| `entity_lifecycle.scanner` | `LifecycleScanner` | Scans entities and marks outdated ones |
| `entity_lifecycle.status_checker` | `LifecycleStatusChecker` | Checks entity lifecycle status |
| `entity_lifecycle.bundle_form_builder` | `BundleLifecycleFormBuilder` | Builds lifecycle settings for bundle forms |
| `entity_lifecycle.entity_type_resolver` | `EntityTypeResolver` | Resolves enabled entity types and bundles |
| `plugin.manager.lifecycle_condition` | `LifecycleConditionManager` | Manages condition plugins |

### Config Entities

- `lifecycle_status`: Defines available lifecycle statuses with properties:
  - `id`: Machine name
  - `label`: Human-readable label
  - `description`: Status description
  - `color`: CSS color class (success, warning, danger, info)
  - `weight`: Sort order
  - `is_default`: Whether this is the default status for new content
  - `requires_review`: Whether this status triggers the review banner

### Base Fields

Added to nodes, media, and user entities via `hook_entity_base_field_info()`:

| Field | Type | Description |
|-------|------|-------------|
| `lifecycle_status` | `list_string` | Current lifecycle status |
| `lifecycle_last_reviewed` | `timestamp` | When content was last reviewed |
| `lifecycle_exclude` | `boolean` | Exclude from scanning |
| `lifecycle_override_days` | `integer` | Custom review period |

### Bundleless Entity Types

Some entity types like `user` don't have bundles. These are handled differently:

- Configuration stored in `entity_lifecycle.settings` under `bundleless_entity_types`
- Enabled via checkbox on main Entity Lifecycle settings form
- Detailed configuration on entity-specific settings pages (e.g.,
  `/admin/config/people/accounts` for users)

## Programmatic Usage

### Scanner Service

```php
// Get the scanner service.
$scanner = \Drupal::service('entity_lifecycle.scanner');

// Scan and mark outdated content.
$results = $scanner->scanAndMark();

// Dry run (don't update entities).
$results = $scanner->scanAndMark(dry_run: TRUE);

// Scan specific entity type/bundle.
$results = $scanner->scanAndMark(
  entity_type: 'node',
  bundle: 'article',
  dry_run: FALSE
);

// Get statistics.
$stats = $scanner->getStatistics();
```

### Status Checker Service

```php
// Get the status checker service.
$checker = \Drupal::service('entity_lifecycle.status_checker');

// Check if entity needs review banner.
if ($checker->needsReviewBanner($entity)) {
  // Display banner.
}

// Check if scanning is enabled for a bundle.
if ($checker->isBundleEnabled('node', 'article')) {
  // Bundle has lifecycle scanning enabled.
}
```

### Helper Functions

```php
// Get all status options as id => label array.
$options = entity_lifecycle_get_status_options();

// Get the default status ID.
$default = entity_lifecycle_get_default_status();

// Get IDs of statuses that require review.
$review_statuses = entity_lifecycle_get_review_statuses();
```

## Condition Plugin System

The module uses a plugin system for defining lifecycle conditions. Conditions
determine when content should be assigned a particular status.

### Plugin Architecture

- **Annotation**: `@LifecycleCondition`
- **Interface**: `LifecycleConditionInterface`
- **Base class**: `LifecycleConditionBase`
- **Manager**: `LifecycleConditionManager`

### Creating a Custom Condition Plugin

1. Create a plugin class in `src/Plugin/LifecycleCondition/`:

```php
<?php

namespace Drupal\my_module\Plugin\LifecycleCondition;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\entity_lifecycle\LifecycleConditionBase;

/**
 * Condition based on custom criteria.
 *
 * @LifecycleCondition(
 *   id = "my_condition",
 *   label = @Translation("My Custom Condition"),
 *   description = @Translation("Description of what this condition checks."),
 *   weight = 50
 * )
 */
class MyCondition extends LifecycleConditionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'threshold' => 10,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(ContentEntityInterface $entity): bool {
    $threshold = $this->conditionConfiguration['threshold'] ?? 10;
    $value = $this->getValue($entity);

    return $value >= $threshold;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ContentEntityInterface $entity): mixed {
    // Return the value being evaluated.
    return $entity->get('some_field')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, array $configuration): array {
    $form['threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Threshold'),
      '#default_value' => $configuration['threshold'] ?? 10,
      '#min' => 0,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function summary(): string {
    $threshold = $this->conditionConfiguration['threshold'] ?? 10;
    return (string) $this->t('value >= @threshold', ['@threshold' => $threshold]);
  }

}
```

2. If your plugin needs services, implement dependency injection:

```php
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

class MyCondition extends LifecycleConditionBase {

  use DependencySerializationTrait;

  protected SomeService $someService;

  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->someService = $container->get('some.service');
    return $instance;
  }

}
```

### Annotation Properties

| Property | Type | Required | Description |
|----------|------|----------|-------------|
| `id` | string | Yes | Unique plugin ID |
| `label` | Translation | Yes | Human-readable label |
| `description` | Translation | No | Plugin description |
| `weight` | int | No | Sort order (default: 0) |
| `entity_types` | array | No | Restrict to specific entity types |

### Interface Methods

| Method | Description |
|--------|-------------|
| `evaluate(ContentEntityInterface $entity): bool` | Returns TRUE if condition is met |
| `getValue(ContentEntityInterface $entity): mixed` | Returns the evaluated value |
| `getFormattedValue(ContentEntityInterface $entity): string` | Returns human-readable value |
| `buildConfigurationForm(array $form, array $configuration): array` | Returns form elements |
| `summary(): string` | Returns summary of current configuration |
| `defaultConfiguration(): array` | Returns default configuration values |
| `isApplicable(string $entity_type_id): bool` | Whether plugin applies to entity type |

## Condition Configuration Structure

Conditions use a groups structure with AND/OR operators between groups:

```yaml
conditions:
  - status: needs_review
    weight: -48
    group_operator: OR  # 'AND' or 'OR'
    groups:
      # Group 1: Old and unchanged
      - plugins:
          - plugin_id: age
            configuration:
              created_months: '0'
              changed_months: '6'
      # Group 2: Completely unused
      - plugins:
          - plugin_id: unused_entity
            configuration:
              max_usage_count: '0'
```

With `group_operator: OR`, content is marked "needs_review" if **either**:
- It hasn't been changed in 6 months
- OR it's completely unused

With `group_operator: AND`, content would need to match **all** groups.

Within each group, all plugins must still match (implicit AND):

```yaml
conditions:
  - status: critical
    weight: -45
    group_operator: OR
    groups:
      # Group 1: Old AND broken links
      - plugins:
          - plugin_id: age
            configuration:
              changed_months: '12'
          - plugin_id: linkchecker
            configuration:
              min_broken_links: '5'
      # Group 2: Completely unused for a long time
      - plugins:
          - plugin_id: age
            configuration:
              changed_months: '24'
          - plugin_id: unused_entity
            configuration:
              max_usage_count: '0'
```

This marks content "critical" if:
- (Unchanged for 12+ months AND has 5+ broken links)
- OR (Unchanged for 24+ months AND completely unused)

## Hooks

### Implemented Hooks

| Hook | Purpose |
|------|---------|
| `hook_entity_base_field_info()` | Adds lifecycle fields to node/media |
| `hook_entity_presave()` | Updates review timestamp on status change |
| `hook_page_top()` | Displays review banner |
| `hook_cron()` | Runs scan at configurable interval (default: daily) |
| `hook_views_data_alter()` | Registers custom Views filter |
| `hook_form_node_type_form_alter()` | Adds lifecycle settings to node type form |
| `hook_form_media_type_form_alter()` | Adds lifecycle settings to media type form |
| `hook_form_alter()` | Groups lifecycle fields in entity forms |
| `hook_theme()` | Defines banner template |

### Extension Hooks for Bundleless Entity Types

The module provides hooks for submodules to register and customize bundleless
entity type support (entity types without bundles, like `user`).

```php
/**
 * Implements hook_entity_lifecycle_bundleless_entity_types().
 *
 * Register bundleless entity types for lifecycle tracking.
 * This hook allows submodules to extend lifecycle tracking to entity types
 * that don't have bundles (like user, file, etc.).
 *
 * @return string[]
 *   Array of entity type IDs.
 */
function my_module_entity_lifecycle_bundleless_entity_types(): array {
  return ['user', 'my_custom_entity'];
}

/**
 * Implements hook_entity_lifecycle_bundleless_query_alter().
 *
 * Alter the entity query for bundleless entity type scanning.
 * Use this to add entity-type-specific conditions to the scan query.
 *
 * @param \Drupal\Core\Entity\Query\QueryInterface $query
 *   The entity query.
 * @param string $entity_type_id
 *   The entity type ID being scanned.
 * @param array $settings
 *   The bundleless entity type settings from config.
 */
function my_module_entity_lifecycle_bundleless_query_alter(
  QueryInterface $query,
  string $entity_type_id,
  array $settings
): void {
  if ($entity_type_id === 'user') {
    // Exclude anonymous user from scanning.
    $query->condition('uid', 0, '>');
  }
}

/**
 * Implements hook_entity_lifecycle_scan_result_alter().
 *
 * Alter scan result data for an entity before it's returned.
 * Use this to add entity-type-specific data to scan results.
 *
 * @param array $data
 *   The result data array (modifiable).
 * @param \Drupal\Core\Entity\ContentEntityInterface $entity
 *   The entity being scanned.
 * @param string $entity_type_id
 *   The entity type ID.
 */
function my_module_entity_lifecycle_scan_result_alter(
  array &$data,
  ContentEntityInterface $entity,
  string $entity_type_id
): void {
  if ($entity_type_id === 'user' && $entity instanceof UserInterface) {
    // Add user-specific display name.
    $data['label'] = $entity->getDisplayName();
    // Add last login info.
    $login = (int) $entity->getLastLoginTime();
    $data['last_login'] = $login
      ? \Drupal::service('date.formatter')->format($login, 'short')
      : t('Never');
  }
}

/**
 * Implements hook_entity_lifecycle_stats_query_alter().
 *
 * Alter the statistics query for bundleless entity types.
 * Use this to add entity-type-specific conditions to stats queries.
 *
 * @param \Drupal\Core\Database\Query\SelectInterface $query
 *   The database select query.
 * @param string $entity_type_id
 *   The entity type ID.
 */
function my_module_entity_lifecycle_stats_query_alter(
  SelectInterface $query,
  string $entity_type_id
): void {
  if ($entity_type_id === 'user') {
    // Exclude anonymous user from statistics.
    $query->condition('e.uid', 0, '>');
  }
}

/**
 * Implements hook_entity_lifecycle_condition_excluded_entity_types().
 *
 * Declares entity types that should not use a specific condition plugin.
 * Use this to exclude certain entity types from using a condition plugin.
 *
 * @param string $condition_plugin_id
 *   The condition plugin ID (e.g., 'age', 'last_login').
 *
 * @return array
 *   Array of entity type IDs that should NOT use this condition plugin.
 */
function my_module_entity_lifecycle_condition_excluded_entity_types(
  string $condition_plugin_id
): array {
  if ($condition_plugin_id === 'age') {
    // The 'age' condition uses 'changed' field which represents profile
    // updates for users, not user activity. Use last_login instead.
    return ['user'];
  }
  return [];
}
```

### Alter Hooks

```php
/**
 * Implements hook_lifecycle_condition_info_alter().
 *
 * Alter condition plugin definitions.
 */
function my_module_lifecycle_condition_info_alter(array &$definitions) {
  // Modify existing condition.
  if (isset($definitions['age'])) {
    $definitions['age']['weight'] = -10;
  }
}
```

## Views Integration

### Custom Filter Plugin

The `LifecycleStatusFilter` extends `InOperator` and dynamically loads status
options from config entities.

Usage in Views:
1. Add filter on `lifecycle_status` field
2. The filter automatically uses `lifecycle_status_filter` plugin
3. Options are loaded from `lifecycle_status` config entities

### Adding to Custom Views

```yaml
# In your_module.views.inc or view config
filters:
  lifecycle_status:
    id: lifecycle_status
    table: node_field_data
    field: lifecycle_status
    plugin_id: lifecycle_status_filter
    exposed: true
```

## Configuration Schema

The module defines schemas in `config/schema/entity_lifecycle.schema.yml`:

- `entity_lifecycle.settings`: Global settings
- `entity_lifecycle.status.*`: Status config entities
- `entity_lifecycle.bundle_settings`: Per-bundle settings
- `entity_lifecycle.condition`: Condition configuration
- `entity_lifecycle.condition_plugin`: Plugin configuration

## File Structure

```
entity_lifecycle/
├── config/
│   ├── install/           # Default configuration
│   ├── optional/          # Optional views
│   └── schema/            # Config schemas
├── css/                   # Stylesheets
├── modules/               # Submodules
│   ├── entity_lifecycle_linkchecker/
│   ├── entity_lifecycle_radioactivity/
│   └── entity_lifecycle_user/
├── src/
│   ├── Annotation/        # Plugin annotations
│   ├── Drush/Commands/    # Drush commands
│   ├── Entity/            # Config entities
│   ├── Form/              # Form classes
│   ├── Plugin/            # Plugins (conditions, views)
│   ├── Service/           # Services
│   ├── LifecycleConditionBase.php
│   ├── LifecycleConditionInterface.php
│   ├── LifecycleConditionManager.php
│   └── LifecycleStatusListBuilder.php
├── entity_lifecycle.info.yml
├── entity_lifecycle.install
├── entity_lifecycle.module
├── entity_lifecycle.permissions.yml
├── entity_lifecycle.routing.yml
└── entity_lifecycle.services.yml
```

## Creating a Bundleless Entity Type Submodule

The `entity_lifecycle_user` submodule serves as a reference implementation for
adding lifecycle support to bundleless entity types. To create support for
another bundleless entity type:

1. Implement `hook_entity_lifecycle_bundleless_entity_types()` to register your
   entity type
2. Implement `hook_entity_base_field_info()` to add lifecycle fields
3. Optionally implement query/result alter hooks for entity-specific behavior
4. Optionally implement `hook_entity_lifecycle_condition_excluded_entity_types()`
   to exclude inapplicable conditions
5. Create entity-specific condition plugins if needed

See `modules/entity_lifecycle_user/` for a complete example.

## Testing

### Running Tests

```bash
# PHPUnit tests
vendor/bin/phpunit modules/custom/entity_lifecycle

# With coverage
vendor/bin/phpunit --coverage-html coverage modules/custom/entity_lifecycle
```

### Code Quality

```bash
# Coding standards
vendor/bin/phpcs --standard=Drupal,DrupalPractice modules/custom/entity_lifecycle

# Static analysis
vendor/bin/phpstan analyse modules/custom/entity_lifecycle --level=5
```

## Contributing

1. Follow Drupal coding standards
2. Add PHPDoc comments to all classes and methods
3. Write tests for new functionality
4. Update documentation as needed
