<?php

namespace Drupal\entity_lifecycle\Form;

use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\entity_lifecycle\Service\EntityTypeResolver;
use Drupal\entity_lifecycle\Service\LifecycleScanner;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Entity Lifecycle settings.
 */
class EntityLifecycleSettingsForm extends ConfigFormBase {

  /**
   * The entity type resolver service.
   */
  protected EntityTypeResolver $entityTypeResolver;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The lifecycle scanner service.
   */
  protected LifecycleScanner $scanner;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeResolver = $container->get('entity_lifecycle.entity_type_resolver');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->scanner = $container->get('entity_lifecycle.scanner');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['entity_lifecycle.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_lifecycle_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('entity_lifecycle.settings');

    $form['cron_interval'] = [
      '#type' => 'select',
      '#title' => $this->t('Automatic scan interval'),
      '#description' => $this->t('How often should the lifecycle scanner run during cron? The scan runs on the first cron execution after the interval has passed. Time-based conditions (like content age) are evaluated during these scans.'),
      '#options' => [
        0 => $this->t('Disabled'),
        -1 => $this->t('Every cron run'),
        6 => $this->t('Every 6 hours'),
        12 => $this->t('Every 12 hours'),
        24 => $this->t('Daily'),
        48 => $this->t('Every 2 days'),
        168 => $this->t('Weekly'),
      ],
      '#default_value' => $config->get('cron_interval') ?? 24,
    ];

    $form['display_banner'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display banner on content needing review'),
      '#description' => $this->t('Show a banner notification when viewing content that needs review.'),
      '#default_value' => $config->get('display_banner') ?? TRUE,
    ];

    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      // Skip anonymous role.
      if ($role->id() !== 'anonymous') {
        $role_options[$role->id()] = $role->label();
      }
    }

    $form['banner_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Banner visibility roles'),
      '#description' => $this->t('Select roles that will see the review banner. Leave empty to show to all users with "View entity lifecycle dashboard" permission.'),
      '#options' => $role_options,
      '#default_value' => $config->get('banner_roles') ?? [],
      '#states' => [
        'visible' => [
          ':input[name="display_banner"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Entity type selection.
    $form['enabled_entity_types_wrapper'] = [
      '#type' => 'details',
      '#title' => $this->t('Enabled entity types'),
      '#open' => TRUE,
      '#description' => $this->t('Select which entity types should support lifecycle tracking. After enabling an entity type here, you can configure individual bundles in their respective settings.'),
    ];

    $supported_types = $this->entityTypeResolver->getSupportedEntityTypes();
    $entity_type_options = [];
    foreach ($supported_types as $entity_type_id => $entity_type) {
      $entity_type_options[$entity_type_id] = $entity_type->getLabel();
    }

    $form['enabled_entity_types_wrapper']['enabled_entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Entity types'),
      '#options' => $entity_type_options,
      '#default_value' => $config->get('enabled_entity_types') ?? ['node', 'media'],
      '#description' => $this->t('Lifecycle fields will be automatically added when entity types are enabled. Clear cache after making changes.'),
    ];

    $form['entity_types_info'] = [
      '#type' => 'details',
      '#title' => $this->t('Configured entity types'),
      '#open' => TRUE,
    ];

    $entity_types = $config->get('entity_types') ?? [];
    $bundleless_entity_types = $config->get('bundleless_entity_types') ?? [];

    if (empty($entity_types) && empty($bundleless_entity_types)) {
      $form['entity_types_info']['message'] = [
        '#markup' => $this->t('<p>No content types are configured for lifecycle scanning yet.</p><p>To enable scanning for a content type, edit the content type and configure the <strong>Lifecycle settings</strong>.</p>'),
      ];
    }
    else {
      // Add batch operation buttons.
      $form['entity_types_info']['actions'] = [
        '#type' => 'actions',
        '#weight' => -10,
      ];

      $form['entity_types_info']['actions']['scan_all'] = [
        '#type' => 'submit',
        '#value' => $this->t('Scan All'),
        '#submit' => ['::scanAllSubmit'],
        '#button_type' => 'primary',
      ];

      $form['entity_types_info']['actions']['rebuild_all'] = [
        '#type' => 'submit',
        '#value' => $this->t('Rebuild All'),
        '#submit' => ['::rebuildAllSubmit'],
      ];

      // Build table with operations.
      $form['entity_types_info']['table'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Entity Type'),
          $this->t('Bundle'),
          $this->t('Review Validity'),
          $this->t('Allow Override'),
          $this->t('Operations'),
        ],
        '#empty' => $this->t('No entity types configured.'),
      ];

      $row_index = 0;

      // Add bundled entity types (e.g., node:news, media:image).
      foreach ($entity_types as $entity_type_id => $bundles) {
        foreach ($bundles as $bundle_id => $settings) {
          $form['entity_types_info']['table'][$row_index]['entity_type'] = [
            '#plain_text' => ucfirst($entity_type_id),
          ];

          $form['entity_types_info']['table'][$row_index]['bundle'] = [
            '#plain_text' => $bundle_id,
          ];

          $review_validity = $settings['review_validity_months'] ?? 12;
          $form['entity_types_info']['table'][$row_index]['review_validity'] = [
            '#plain_text' => $this->formatPlural($review_validity, '1 month', '@count months'),
          ];

          $form['entity_types_info']['table'][$row_index]['allow_override'] = [
            '#plain_text' => !empty($settings['allow_override']) ? $this->t('Yes') : $this->t('No'),
          ];

          $form['entity_types_info']['table'][$row_index]['operations'] = [
            '#type' => 'operations',
            '#links' => [
              'scan' => [
                'title' => $this->t('Scan'),
                'url' => Url::fromRoute('entity_lifecycle.scan_bundle', [
                  'entity_type_id' => $entity_type_id,
                  'bundle' => $bundle_id,
                ]),
              ],
              'rebuild' => [
                'title' => $this->t('Rebuild'),
                'url' => Url::fromRoute('entity_lifecycle.rebuild_bundle', [
                  'entity_type_id' => $entity_type_id,
                  'bundle' => $bundle_id,
                ]),
              ],
            ],
          ];

          $row_index++;
        }
      }

      // Add bundleless entity types (e.g., user).
      foreach ($bundleless_entity_types as $entity_type_id => $settings) {
        if (empty($settings['enabled'])) {
          continue;
        }

        $entity_type_definition = $this->entityTypeManager->getDefinition($entity_type_id);

        $form['entity_types_info']['table'][$row_index]['entity_type'] = [
          '#plain_text' => (string) $entity_type_definition->getLabel(),
        ];

        $form['entity_types_info']['table'][$row_index]['bundle'] = [
          '#plain_text' => '-',
        ];

        $review_validity = $settings['review_validity_months'] ?? 12;
        $form['entity_types_info']['table'][$row_index]['review_validity'] = [
          '#plain_text' => $this->formatPlural($review_validity, '1 month', '@count months'),
        ];

        $form['entity_types_info']['table'][$row_index]['allow_override'] = [
          '#plain_text' => !empty($settings['allow_override']) ? $this->t('Yes') : $this->t('No'),
        ];

        $form['entity_types_info']['table'][$row_index]['operations'] = [
          '#type' => 'operations',
          '#links' => [
            'scan' => [
              'title' => $this->t('Scan'),
              'url' => Url::fromRoute('entity_lifecycle.scan_bundle', [
                'entity_type_id' => $entity_type_id,
                'bundle' => $entity_type_id,
              ]),
            ],
            'rebuild' => [
              'title' => $this->t('Rebuild'),
              'url' => Url::fromRoute('entity_lifecycle.rebuild_bundle', [
                'entity_type_id' => $entity_type_id,
                'bundle' => $entity_type_id,
              ]),
            ],
          ],
        ];

        $row_index++;
      }

      $form['entity_types_info']['help'] = [
        '#markup' => $this->t('<p>To modify these settings, edit the individual content types, media types, or entity types.</p>'),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $banner_roles = array_filter($form_state->getValue('banner_roles'));
    $enabled_entity_types = array_filter($form_state->getValue('enabled_entity_types'));

    $this->config('entity_lifecycle.settings')
      ->set('cron_interval', (int) $form_state->getValue('cron_interval'))
      ->set('display_banner', (bool) $form_state->getValue('display_banner'))
      ->set('banner_roles', array_values($banner_roles))
      ->set('enabled_entity_types', array_values($enabled_entity_types))
      ->save();

    // Clear the entity type resolver cache.
    $this->entityTypeResolver->resetCache();

    parent::submitForm($form, $form_state);
  }

  /**
   * Submit handler for "Scan All" button.
   */
  public function scanAllSubmit(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('entity_lifecycle.settings');
    $entity_types = $config->get('entity_types') ?? [];
    $bundleless_entity_types = $config->get('bundleless_entity_types') ?? [];

    $operations = [];

    // Add bundled entity types.
    foreach ($entity_types as $entity_type_id => $bundles) {
      foreach (array_keys($bundles) as $bundle_id) {
        $operations[] = [
          [static::class, 'batchScanOperation'],
          [$entity_type_id, $bundle_id],
        ];
      }
    }

    // Add bundleless entity types.
    foreach ($bundleless_entity_types as $entity_type_id => $settings) {
      if (!empty($settings['enabled'])) {
        $operations[] = [
          [static::class, 'batchScanOperation'],
          [$entity_type_id, $entity_type_id],
        ];
      }
    }

    $batch = [
      'title' => $this->t('Scanning all configured entity types...'),
      'operations' => $operations,
      'finished' => [static::class, 'batchScanFinished'],
      'progress_message' => $this->t('Processed @current of @total bundles.'),
    ];

    batch_set($batch);
  }

  /**
   * Submit handler for "Rebuild All" button.
   */
  public function rebuildAllSubmit(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('entity_lifecycle.settings');
    $entity_types = $config->get('entity_types') ?? [];
    $bundleless_entity_types = $config->get('bundleless_entity_types') ?? [];

    $operations = [];

    // Add bundled entity types.
    foreach ($entity_types as $entity_type_id => $bundles) {
      foreach (array_keys($bundles) as $bundle_id) {
        $operations[] = [
          [static::class, 'batchRebuildOperation'],
          [$entity_type_id, $bundle_id],
        ];
      }
    }

    // Add bundleless entity types.
    foreach ($bundleless_entity_types as $entity_type_id => $settings) {
      if (!empty($settings['enabled'])) {
        $operations[] = [
          [static::class, 'batchRebuildOperation'],
          [$entity_type_id, $entity_type_id],
        ];
      }
    }

    $batch = [
      'title' => $this->t('Rebuilding all configured entity types...'),
      'operations' => $operations,
      'finished' => [static::class, 'batchRebuildFinished'],
      'progress_message' => $this->t('Processed @current of @total bundles.'),
    ];

    batch_set($batch);
  }

  /**
   * Batch operation callback for scanning.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array $context
   *   The batch context.
   */
  public static function batchScanOperation(string $entity_type_id, string $bundle_id, array &$context): void {
    static::initializeBatchContext($context, 'scanned', $entity_type_id, $bundle_id);

    $scanner = \Drupal::service('entity_lifecycle.scanner');
    $result = $scanner->scanAndMark(FALSE, $entity_type_id, $bundle_id);

    static::storeBatchResult(
      $context,
      'scanned',
      $entity_type_id,
      $bundle_id,
      $result['count'] ?? 0,
      'Scanned',
      'Scanning'
    );
  }

  /**
   * Batch operation callback for rebuilding.
   *
   * Uses the LifecycleScanner service for shared rebuild logic.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param array $context
   *   The batch context.
   */
  public static function batchRebuildOperation(string $entity_type_id, string $bundle_id, array &$context): void {
    static::initializeBatchContext($context, 'rebuilt', $entity_type_id, $bundle_id);

    $scanner = \Drupal::service('entity_lifecycle.scanner');
    $result = $scanner->rebuildAndMark(FALSE, $entity_type_id, $bundle_id);

    static::storeBatchResult(
      $context,
      'rebuilt',
      $entity_type_id,
      $bundle_id,
      $result['count'] ?? 0,
      'Rebuilt',
      'Rebuilding'
    );
  }

  /**
   * Batch finished callback for scan operations.
   */
  public static function batchScanFinished(bool $success, array $results, array $operations): void {
    if ($success) {
      $total = array_sum($results['scanned'] ?? []);
      \Drupal::messenger()->addStatus(t('Successfully scanned @count entities across @bundles bundles.', [
        '@count' => $total,
        '@bundles' => count($results['scanned'] ?? []),
      ]));

      // Display detailed messages.
      foreach ($results['messages'] ?? [] as $message) {
        \Drupal::messenger()->addStatus($message);
      }
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during the scan operation.'));
    }
  }

  /**
   * Batch finished callback for rebuild operations.
   */
  public static function batchRebuildFinished(bool $success, array $results, array $operations): void {
    if ($success) {
      $total = array_sum($results['rebuilt'] ?? []);
      \Drupal::messenger()->addStatus(t('Successfully rebuilt @count entities across @bundles bundles.', [
        '@count' => $total,
        '@bundles' => count($results['rebuilt'] ?? []),
      ]));

      // Display detailed messages.
      foreach ($results['messages'] ?? [] as $message) {
        \Drupal::messenger()->addStatus($message);
      }
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during the rebuild operation.'));
    }
  }

  /**
   * Initializes batch context for an operation.
   *
   * @param array $context
   *   The batch context array.
   * @param string $operation_type
   *   The operation type (e.g., 'scanned', 'rebuilt').
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   */
  protected static function initializeBatchContext(array &$context, string $operation_type, string $entity_type_id, string $bundle_id): void {
    // Initialize results array.
    if (!isset($context['results'][$operation_type])) {
      $context['results'][$operation_type] = [];
      $context['results']['messages'] = [];
    }

    // Initialize sandbox for progress tracking.
    if (!isset($context['sandbox']['progress'])) {
      $context['sandbox']['progress'] = 0;
      $context['sandbox']['bundle_key'] = "$entity_type_id:$bundle_id";
      $context['sandbox']['max'] = static::countBundleEntities($entity_type_id, $bundle_id);
    }
  }

  /**
   * Counts entities for a specific bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return int
   *   The number of entities.
   */
  protected static function countBundleEntities(string $entity_type_id, string $bundle_id): int {
    $entity_type_manager = \Drupal::entityTypeManager();
    $storage = $entity_type_manager->getStorage($entity_type_id);
    $query = $storage->getQuery()->accessCheck(FALSE);

    $entity_type_definition = $entity_type_manager->getDefinition($entity_type_id);
    $bundle_key = $entity_type_definition->getKey('bundle');
    if ($bundle_key) {
      $query->condition($bundle_key, $bundle_id);
    }

    return (int) $query->count()->execute();
  }

  /**
   * Stores batch operation results.
   *
   * @param array $context
   *   The batch context array.
   * @param string $operation_type
   *   The operation type (e.g., 'scanned', 'rebuilt').
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param int $count
   *   The number of entities processed.
   * @param string $past_tense
   *   Past tense verb for the operation (e.g., 'Scanned', 'Rebuilt').
   * @param string $present_tense
   *   Present tense verb for the operation (e.g., 'Scanning', 'Rebuilding').
   */
  protected static function storeBatchResult(
    array &$context,
    string $operation_type,
    string $entity_type_id,
    string $bundle_id,
    int $count,
    string $past_tense,
    string $present_tense,
  ): void {
    $key = "$entity_type_id:$bundle_id";

    // Store count result.
    $context['results'][$operation_type][$key] = $count;

    // Add message for display.
    $context['results']['messages'][] = t('@action @type (@bundle): @count entities processed', [
      '@action' => $past_tense,
      '@type' => $entity_type_id,
      '@bundle' => $bundle_id,
      '@count' => $count,
    ]);

    // Set progress message.
    $context['message'] = t('@action @type (@bundle): @count entities', [
      '@action' => $present_tense,
      '@type' => $entity_type_id,
      '@bundle' => $bundle_id,
      '@count' => $count,
    ]);

    // Mark as finished.
    $context['finished'] = 1;
  }

}
