<?php

namespace Drupal\entity_lifecycle\Plugin\views\area;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Displays entity lifecycle status summary statistics.
 *
 * @ViewsArea("lifecycle_summary")
 *
 * @ingroup views_area_handlers
 */
class LifecycleSummary extends AreaPluginBase {

  /**
   * The database connection.
   */
  protected Connection $database;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, Connection $database, EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('entity_type.manager'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['entity_type'] = ['default' => ''];
    $options['empty'] = ['default' => FALSE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // Build entity type options dynamically.
    $options = [
      '' => $this->t('- Auto-detect -'),
      'node' => $this->t('Content'),
      'media' => $this->t('Media'),
    ];

    // Allow modules to add their entity types.
    $this->moduleHandler->alter('entity_lifecycle_summary_entity_types', $options);

    $form['entity_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity type'),
      '#description' => $this->t('Select the entity type to display statistics for. Leave empty to auto-detect from view base table.'),
      '#options' => $options,
      '#default_value' => $this->options['entity_type'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    if (!$empty || !empty($this->options['empty'])) {
      return $this->buildSummary();
    }
    return [];
  }

  /**
   * Builds the summary statistics.
   *
   * @return array
   *   Render array.
   */
  protected function buildSummary(): array {
    // Determine entity type from configuration or view base table.
    $entity_type_id = $this->options['entity_type'];
    if (empty($entity_type_id)) {
      $entity_type_id = $this->detectEntityType();
    }

    if (empty($entity_type_id)) {
      return [];
    }

    // Get entity type definition.
    try {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
    }
    catch (\Exception $e) {
      return [];
    }

    // Get the data table.
    $data_table = $entity_type->getDataTable() ?: $entity_type->getBaseTable();
    if (empty($data_table)) {
      return [];
    }

    // Load status entities to get colors.
    try {
      $status_storage = $this->entityTypeManager->getStorage('lifecycle_status');
      $statuses = $status_storage->loadMultiple();
    }
    catch (\Exception $e) {
      // If status storage is unavailable, return empty result.
      return [];
    }

    if (empty($statuses)) {
      return [];
    }

    // Check if View has active filters and get filtered entity IDs.
    $filtered_ids = $this->getFilteredEntityIds($entity_type);

    // If filters are active but no results match, return empty summary.
    if ($filtered_ids !== NULL && empty($filtered_ids)) {
      return $this->buildEmptySummary($statuses, $entity_type_id);
    }

    // Build query for status counts.
    $query = $this->database->select($data_table, 'e');
    $query->addExpression('COUNT(*)', 'count');
    $query->fields('e', ['lifecycle_status']);

    // If filters are active, limit statistics to filtered entity IDs.
    if ($filtered_ids !== NULL) {
      $id_key = $entity_type->getKey('id');
      $query->condition("e.{$id_key}", $filtered_ids, 'IN');
    }

    // Allow modules to alter the summary query for their entity types.
    $this->moduleHandler->alter('entity_lifecycle_summary_query', $query, $entity_type_id);

    $query->groupBy('e.lifecycle_status');

    $result = $query->execute();
    $counts = [];
    $total = 0;

    foreach ($result as $row) {
      $status = $row->lifecycle_status ?: 'no_status';
      $counts[$status] = (int) $row->count;
      $total += (int) $row->count;
    }

    // Build summary items with color information.
    $items = $this->buildSummaryItems($statuses, $counts);

    return $this->buildRenderArray($items, $total, $entity_type_id);
  }

  /**
   * Gets entity IDs from the View's filtered query.
   *
   * When the View has active exposed filters, this method clones the View,
   * removes paging, and executes to get all matching entity IDs. These IDs
   * are then used to limit the statistics query to only filtered entities.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   *
   * @return array|null
   *   Array of entity IDs if filters are active, NULL if no filters applied.
   */
  protected function getFilteredEntityIds(EntityTypeInterface $entity_type): ?array {
    // Get exposed input from the View.
    $exposed_input = $this->view->getExposedInput();

    // Filter out empty values, paging, and sort parameters.
    // Exposed input can contain strings or arrays (for multi-select filters).
    $active_filters = [];
    foreach ($exposed_input as $key => $value) {
      // Ignore paging and sort parameters.
      if (in_array($key, ['page', 'order', 'sort', 'items_per_page'], TRUE)) {
        continue;
      }
      // Check for empty arrays (multi-value filters).
      if (is_array($value)) {
        if (!empty($value)) {
          $active_filters[$key] = $value;
        }
        continue;
      }
      // Check for non-empty string filter values.
      if ($value !== '' && $value !== 'All') {
        $active_filters[$key] = $value;
      }
    }

    // If no active filters, return NULL to use the full unfiltered query.
    if (empty($active_filters)) {
      return NULL;
    }

    // Clone the View to get all matching entity IDs without paging.
    $view_id = $this->view->id();
    $display_id = $this->view->current_display;

    $view_clone = Views::getView($view_id);
    if (!$view_clone) {
      return NULL;
    }

    $view_clone->setDisplay($display_id);
    $view_clone->setExposedInput($exposed_input);

    // Use 'none' pager to get all results.
    $view_clone->setItemsPerPage(0);

    // Build and execute the View.
    $view_clone->build();

    // Get the entity ID key.
    $id_key = $entity_type->getKey('id');

    // Execute the query and extract entity IDs.
    $entity_ids = [];

    try {
      $view_clone->execute();

      foreach ($view_clone->result as $row) {
        // The entity ID is typically stored in the row with the ID key name.
        if (isset($row->{$id_key})) {
          $entity_ids[] = $row->{$id_key};
        }
        elseif (property_exists($row, '_entity') && $row->_entity !== NULL) {
          // Fallback: get ID from the entity object.
          $entity_ids[] = $row->_entity->id();
        }
      }
    }
    catch (\Exception $e) {
      // If query fails, fall back to unfiltered statistics.
      return NULL;
    }

    return $entity_ids;
  }

  /**
   * Builds an empty summary render array when no results match filters.
   *
   * @param array $statuses
   *   Array of lifecycle status entities.
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   Render array with all counts set to zero.
   */
  protected function buildEmptySummary(array $statuses, string $entity_type_id): array {
    $items = $this->buildSummaryItems($statuses, []);

    return $this->buildRenderArray($items, 0, $entity_type_id);
  }

  /**
   * Builds summary items array from status entities and counts.
   *
   * @param array $statuses
   *   Array of lifecycle status entities.
   * @param array $counts
   *   Associative array of status ID => count.
   *
   * @return array
   *   Array of summary items with label, count, status, and color keys.
   */
  protected function buildSummaryItems(array $statuses, array $counts): array {
    $items = [];
    foreach ($statuses as $status) {
      /** @var \Drupal\entity_lifecycle\Entity\LifecycleStatus $status */
      $status_id = $status->id();
      $items[] = [
        'label' => $status->label(),
        'count' => $counts[$status_id] ?? 0,
        'status' => $status_id,
        'color' => $status->getColor(),
      ];
    }
    // Add "no status" count.
    $items[] = [
      'label' => $this->t('No status'),
      'count' => $counts['no_status'] ?? 0,
      'status' => 'no_status',
      'color' => 'secondary',
    ];

    return $items;
  }

  /**
   * Builds the render array for the summary.
   *
   * @param array $items
   *   Array of summary items.
   * @param int $total
   *   Total count of entities.
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   Render array for the summary.
   */
  protected function buildRenderArray(array $items, int $total, string $entity_type_id): array {
    // Determine which theme to use based on entity type.
    $render_info = [
      'theme' => 'entity_lifecycle_summary',
      'library' => 'entity_lifecycle/admin',
      'entity_label' => $this->t('items'),
    ];

    // Allow modules to override theme/library for their entity types.
    $this->moduleHandler->alter('entity_lifecycle_summary_render', $render_info, $entity_type_id);

    return [
      '#theme' => $render_info['theme'],
      '#items' => $items,
      '#total' => $total,
      '#entity_type' => $entity_type_id,
      '#entity_label' => $render_info['entity_label'],
      '#attached' => [
        'library' => [$render_info['library']],
      ],
    ];
  }

  /**
   * Detects entity type from view base table.
   *
   * @return string|null
   *   The entity type ID or NULL if not detected.
   */
  protected function detectEntityType(): ?string {
    $base_table = $this->view->storage->get('base_table');

    // Default mapping for core entity types.
    $map = [
      'node_field_data' => 'node',
      'node' => 'node',
      'media_field_data' => 'media',
      'media' => 'media',
    ];

    // Allow modules to add their base table mappings.
    $this->moduleHandler->alter('entity_lifecycle_summary_base_tables', $map);

    return $map[$base_table] ?? NULL;
  }

}
