<?php

namespace Drupal\entity_logger;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LogMessageParserInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\entity_logger\Entity\EntityLogEntryInterface;
use Drupal\entity_logger\Event\EntityLoggerAvailableEntityTypesEvent;
use Drupal\entity_logger\Event\EntityLoggerEvents;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Service for logging to entities.
 */
class EntityLogger implements EntityLoggerInterface {

  /**
   * EntityLogger constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\Core\Logger\LogMessageParserInterface $parser
   *   The parser to use when extracting message variables.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected LogMessageParserInterface $parser,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function log(EntityInterface $entity, string $message, array $context = [], int $severity = RfcLogLevel::INFO, ?string $logger_channel = NULL): ?EntityLogEntryInterface {
    $entityLoggerSettings = $this->configFactory->get('entity_logger.settings');
    $enabled_entity_types = $entityLoggerSettings->get('enabled_entity_types');
    if (!in_array($entity->getEntityTypeId(), $enabled_entity_types, TRUE)) {
      return NULL;
    }

    // Convert PSR3-style messages to \Drupal\Component\Render\FormattableMarkup
    // style, so they can be translated too in runtime.
    $context = $this->parser->parseMessagePlaceholders($message, $context);

    /** @var \Drupal\entity_logger\Entity\EntityLogEntryInterface $log_entry */
    $entityLogEntryStorage = $this->entityTypeManager->getStorage('entity_log_entry');
    $log_entry = $entityLogEntryStorage->create([]);
    $log_entry->setTargetEntity($entity);
    $log_entry->setMessage($message, $context);
    $log_entry->setSeverity($severity);
    $log_entry->save();

    if ($logger_channel) {
      $this->loggerFactory->get($logger_channel)->log($severity, $message, $context);
    }
    return $log_entry;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableEntityTypesForLogging(): array {
    // Get entity types that are explicitly made available via the event.
    $event_entity_types = [];
    $event = new EntityLoggerAvailableEntityTypesEvent($event_entity_types);
    $this->eventDispatcher->dispatch($event, EntityLoggerEvents::AVAILABLE_ENTITY_TYPES);
    $event_entity_types = $event->getEntityTypes();

    // Render a list of all entity types that should be available.
    $entity_types = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      if ($entity_type->hasLinkTemplate('canonical') || in_array($entity_type_id, $event_entity_types, TRUE)) {
        $entity_types[$entity_type_id] = $entity_type->getLabel();
      }
    }
    return $entity_types;
  }

}
