<?php

namespace Drupal\entity_logger\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\entity_logger\EntityLoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Entity Logger settings.
 */
class EntityLoggerSettingsForm extends ConfigFormBase {

  /**
   * Constructs an EntityLoggerSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Defines the configuration object factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\entity_logger\EntityLoggerInterface $entityLogger
   *   The entity logger service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityLoggerInterface $entityLogger,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_logger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_logger_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['entity_logger.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('entity_logger.settings');

    $form['enabled_entity_types'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Enabled entity types'),
      '#description' => $this->t('Enable logging for these entity types.'),
      '#tree' => TRUE,
    ];

    // Get all applicable entity types.
    foreach ($this->entityLogger->getAvailableEntityTypesForLogging() as $entity_type_id => $entity_type_name) {
      $form['enabled_entity_types'][$entity_type_id] = [
        '#type' => 'checkbox',
        '#title' => $entity_type_name,
        '#default_value' => in_array($entity_type_id, $config->get('enabled_entity_types')),
      ];
    }

    // Retention period options in days.
    $retention_options = [
      0 => $this->t('Never delete'),
      7 => $this->t('1 week'),
      30 => $this->t('1 month'),
      90 => $this->t('3 months'),
      180 => $this->t('6 months'),
      365 => $this->t('1 year'),
    ];

    $form['retention_period'] = [
      '#type' => 'select',
      '#title' => $this->t('Retention period'),
      '#description' => $this->t('Specify how long to keep log entries. Entries older than the specified period will be deleted in batch on cron.'),
      '#options' => $retention_options,
      '#default_value' => $config->get('retention_period') ?? 90,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('entity_logger.settings');
    $enabled_entity_types = array_filter($form_state->getValue('enabled_entity_types'), static function ($checked) {
      return (bool) $checked;
    });
    $config->set('enabled_entity_types', array_keys($enabled_entity_types));

    // Save retention period.
    $config->set('retention_period', (int) $form_state->getValue('retention_period'));

    // Save the config.
    $config->save();

    parent::submitForm($form, $form_state);
  }

}
