<?php

declare(strict_types=1);

namespace Drupal\entity_logger\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Psr\Log\LoggerInterface;

/**
 * Provides cron hooks.
 */
class CronHooks {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected TimeInterface $time,
    protected LoggerInterface $logger,
  ) {}

  /**
   * Implements hook_cron().
   *
   * Cleans up old entity log entries based on the configured retention period.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  #[Hook('cron')]
  public function cron(): void {
    $config = $this->configFactory->get('entity_logger.settings');
    $retentionDays = (int) $config->get('retention_period');
    if ($retentionDays > 0) {
      $cutoff = $this->time->getRequestTime() - ($retentionDays * 86400);
      /** @var \Drupal\entity_logger\EntityLogEntryStorageInterface $storage */
      $storage = $this->entityTypeManager->getStorage('entity_log_entry');

      $totalDeleted = 0;
      $batchSize = 100;
      $startTime = microtime(TRUE);
      $timeLimit = 10;

      do {
        // Check if we've exceeded the time limit and emit a warning if we have.
        // Especially when enabling this after a long period of time, it is fine
        // to exceed the time limit for several runs, but when this happens
        // consistently, it is a sign that the log fills faster than we can
        // clean it up. Consider running cron more often, or reduce the amount
        // of logging.
        if ((microtime(TRUE) - $startTime) > $timeLimit) {
          $this->logger->warning(
            'Retention cleanup stopped after @seconds seconds to avoid using excessive time. Deleted @count entries older than @days days.',
            [
              '@seconds' => $timeLimit,
              '@count' => $totalDeleted,
              '@days' => $retentionDays,
            ],
          );
          return;
        }

        // Find entries older than the cutoff timestamp, limited to batch size.
        $ids = $storage->getQuery()
          ->accessCheck(FALSE)
          ->condition('created', $cutoff, '<')
          ->sort('id')
          ->range(0, $batchSize)
          ->execute();

        if ($ids) {
          $entries = $storage->loadMultiple($ids);
          $storage->delete($entries);
          $deleted = count($entries);
          $totalDeleted += $deleted;

          // Clear the entity storage cache to free memory.
          $storage->resetCache($ids);
        }
      } while (!empty($ids));

      if ($totalDeleted > 0) {
        $this->logger->notice(
          'Deleted @count log entries older than @days days.',
          ['@count' => $totalDeleted, '@days' => $retentionDays]
        );
      }
    }
  }

}
