<?php

declare(strict_types=1);

namespace Drupal\entity_logger\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides entity hooks.
 */
class EntityHooks {

  use StringTranslationTrait;

  /**
   * Constructs an EntityHooks object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountInterface $currentUser,
  ) {}

  /**
   * Implements hook_entity_type_alter().
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface[] $entityTypes
   *   The master entity type list to alter.
   */
  #[Hook('entity_type_alter')]
  public function entityTypeAlter(array &$entityTypes): void {
    $entity_logger_settings = $this->configFactory->get('entity_logger.settings');
    $enabledEntityTypes = $entity_logger_settings->get('enabled_entity_types');
    if (is_array($enabledEntityTypes)) {
      foreach ($entityTypes as $entityTypeId => $entityType) {
        if (in_array($entityTypeId, $enabledEntityTypes, TRUE)) {
          $entityType->setLinkTemplate('entity-logger', "/entity_logger/$entityTypeId/{{$entityTypeId}}");
        }
      }
    }
  }

  /**
   * Implements hook_entity_predelete().
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The given entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  #[Hook('entity_predelete')]
  public function entityPreDelete(EntityInterface $entity): void {
    /** @var \Drupal\entity_logger\EntityLogEntryStorageInterface $log_entry_storage */
    $log_entry_storage = $this->entityTypeManager->getStorage('entity_log_entry');
    $log_entry_storage->deleteForTargetEntity($entity);
  }

  /**
   * Implements hook_entity_operation().
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity on which to define an operation.
   *
   * @return array
   *   An array of operation definitions.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  #[Hook('entity_operation')]
  public function entityOperation(EntityInterface $entity): array {
    $operations = [];
    if ($this->currentUser->hasPermission('view entity log entries')) {
      if ($entity->hasLinkTemplate('entity-logger')) {
        $operations['entity_logger'] = [
          'title' => $this->t('Log'),
          'weight' => 50,
          'url' => $entity->toUrl('entity-logger'),
        ];
      }
    }
    return $operations;
  }

}
