<?php

namespace Drupal\entity_mesh\Batches;

use Drupal\Core\Batch\BatchBuilder;

/**
 * General batch operations.
 */
class GeneralBatch {

  /**
   * Get batch object preconfigured.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   Batch object.
   */
  public static function getBatchObjectPreconfigured() {
    $batch = new BatchBuilder();
    $batch->setTitle('Running Entity Mesh batch process.')
      ->setFinishCallback([self::class, 'batchFinished'])
      ->setInitMessage('Commencing')
      ->setProgressMessage('Processing...')
      ->setErrorMessage('An error occurred during processing.');
    return $batch;
  }

  /**
   * Batch finished.
   *
   * @param bool $success
   *   Success.
   * @param array $results
   *   Results.
   * @param array $operations
   *   Operations.
   * @param string $elapsed
   *   Elapsed time.
   *
   * @SuppressWarnings(PHPMD.UnusedFormalParameter)
   */
  public static function batchFinished(bool $success, array $results, array $operations, string $elapsed) {
    if ($success) {
      $message = \Drupal::translation()->formatPlural(
        count($results),
        'One item processed in @elapsed.',
        '@count items processed in @elapsed.',
        ['@elapsed' => $elapsed]
      );
      \Drupal::messenger()->addMessage($message);
    }
    else {
      $message = t('Finished with an error.');
      \Drupal::messenger()->addError($message);
    }
  }

  /**
   * Generate batch.
   *
   * @param array $types
   *   Mesh types.
   *
   * @return \Drupal\Core\Batch\BatchBuilder
   *   Batch object.
   */
  public static function generateBatch($types) {
    $mesh_type_configuration =
      [
        'node' => NodeBatch::class,
        // 'menu' => MenuBatch::class,
      ];

    $batch = GeneralBatch::getBatchObjectPreconfigured();
    $grouped_items = [];
    $total = 0;

    foreach ($mesh_type_configuration as $mesh_type => $batch_class) {
      if (!in_array($mesh_type, $types)) {
        continue;
      }
      $grouped_items[$mesh_type] = $batch_class::getItemsForOperation();
      $total += count($grouped_items[$mesh_type]);
    }

    $count = 1;
    foreach ($grouped_items as $mesh_type => $group) {
      $batch_class = $mesh_type_configuration[$mesh_type];
      $batch_class::clearEntityMesh();
      foreach ($group as $item) {
        $args = [$item, $total, $count];
        $batch->addOperation([$batch_class, 'operation'], $args);
        $count++;
      }
    }

    return $batch;
  }

}
