<?php

namespace Drupal\entity_mesh;

use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class Target to instance target objects.
 *
 * @package Drupal\entity_mesh
 */
class Target implements TargetInterface {

  use HelperTrait;

  /**
   * Category.
   *
   * @var string|null
   */
  protected $category;

  /**
   * Category.
   *
   * @var string|null
   */
  protected $subCategory;

  /**
   * Hash ID.
   *
   * @var string|null
   */
  protected $hashId;

  /**
   * Current domain.
   *
   * @var string
   */
  protected $currentDomain = '';

  /**
   * Base path.
   *
   * @var string
   */
  protected $basePath = '';

  /**
   * Href property.
   *
   * @var string|null
   */
  protected $targetHref;

  /**
   * Path property.
   *
   * @var string|null
   */
  protected $targetPath;

  /**
   * Scheme property.
   *
   * @var string|null
   */
  protected $targetScheme;

  /**
   * Link type property.
   *
   * @var string|null
   */
  protected $targetLinkType;

  /**
   * Entity Type.
   *
   * @var string|null
   */
  protected $targetEntityType;

  /**
   * Entity Bundle.
   *
   * @var string|null
   */
  protected $targetEntityBundle;

  /**
   * Entity ID.
   *
   * @var string|null
   */
  protected $targetEntityId;

  /**
   * Langcode.
   *
   * @var string|null
   */
  protected $targetEntityLangcode;

  /**
   * Title.
   *
   * @var string|null
   */
  protected $title;

  /**
   * Whether self domain URLs should be considered internal.
   *
   * @var bool
   */
  protected $selfDomainInternal;

  /**
   * {@inheritdoc}
   */
  public static function create(RequestStack $request_stack, bool $self_domain_internal = TRUE): Target {
    $target = new self();
    $current_request = $request_stack->getCurrentRequest();
    if ($current_request) {
      $target->currentDomain = $current_request->getHost();
      $target->basePath = $current_request->getBasePath();
    }
    $target->selfDomainInternal = $self_domain_internal;
    return $target;
  }

  /**
   * {@inheritdoc}
   */
  public function getCategory(): ?string {
    return $this->category ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setCategory($category) {
    $this->category = $category;
  }

  /**
   * {@inheritdoc}
   */
  public function getSubcategory(): ?string {
    return $this->subCategory ?? $this->getCategory();
  }

  /**
   * {@inheritdoc}
   */
  public function setSubcategory($sub_category) {
    $this->subCategory = $sub_category;
  }

  /**
   * {@inheritdoc}
   */
  public function getHref(): ?string {
    return $this->targetHref ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setHref(?string $href) {
    $this->targetHref = $href;
  }

  /**
   * {@inheritdoc}
   */
  public function processHrefAndSetComponents(string $href) {
    $this->setHref($this->sanitizeUrl($href));
    $href_components = parse_url($href);
    if (!is_array($href_components)) {
      return;
    }

    $this->setIfExternalOrInternal($href_components);

    // If is internal, we need to set the path.
    if ($this->getLinkType() === 'internal') {
      $path = $href_components['path'] ?? '';
      if (!empty($path)) {
        $path = $this->sanitizeUrl($path);

        // Remove base path from the URL if Drupal is installed
        // in a subdirectory.
        if (!empty($this->basePath) && strpos($path, $this->basePath) === 0) {
          $path = substr($path, strlen($this->basePath));
          // Ensure path starts with /.
          $path = '/' . ltrim($path, '/');
        }
      }
      $this->setPath($path);
    }
  }

  /**
   * Set if the target is external or internal.
   *
   * @param array $href_components
   *   Href components, result of parse_url.
   */
  protected function setIfExternalOrInternal(array $href_components) {
    // This is internal because the relative path has not defined schema.
    if (empty($href_components['scheme'])) {
      $this->setLinkType('internal');
      return;
    }

    $this->setScheme($href_components['scheme']);

    // Cover the case of the href being internal but absolute.
    if ($this->selfDomainInternal &&
      in_array($href_components['scheme'], ['http', 'https']) &&
      $href_components['host'] === $this->currentDomain
        ) {
      $this->setLinkType('internal');
      return;
    }

    $this->setLinkType('external');
  }

  /**
   * {@inheritdoc}
   */
  public function getScheme(): ?string {
    return $this->targetScheme ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setScheme(?string $scheme) {
    $this->targetScheme = $scheme;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityType(): ?string {
    return $this->targetEntityType ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityType(?string $entity_type) {
    $this->targetEntityType = $entity_type;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityId(): ?string {
    return $this->targetEntityId ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityId(?string $entity_id) {
    $this->targetEntityId = $entity_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityLangcode(): string {
    return $this->targetEntityLangcode ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityLangcode(?string $entity_langcode) {
    $this->targetEntityLangcode = $entity_langcode;
  }

  /**
   * {@inheritdoc}
   */
  public function getLinkType(): ?string {
    return $this->targetLinkType ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setLinkType(?string $link_type) {
    return $this->targetLinkType = $link_type;
  }

  /**
   * {@inheritdoc}
   */
  public function getPath(): ?string {
    return $this->targetPath ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setPath(?string $path) {
    $this->targetPath = $path;
  }

  /**
   * {@inheritdoc}
   */
  public function getHashId(): ?string {
    return $this->hashId ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityBundle(): ?string {
    return $this->targetEntityBundle ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityBundle(string $entity_bundle) {
    $this->targetEntityBundle = $entity_bundle;
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle(): ?string {
    return $this->title ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setTitle(string $title) {
    $this->title = $title;
  }

  /**
   * {@inheritdoc}
   */
  public function setHashId() {
    $string_to_hash = '';
    if ($this->getEntityType() && $this->getEntityId()) {
      $string_to_hash = $this->getEntityType() . $this->getEntityId();
    }
    elseif ($this->getHref()) {
      $string_to_hash = $this->getHref();
    }

    if ($string_to_hash) {
      $this->hashId = $this->generateHash((string) $string_to_hash);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function toArray(): array {
    return [
      'category' => $this->getCategory(),
      'subcategory' => $this->getSubcategory(),
      'target_link_type' => $this->getLinkType(),
      'target_href' => $this->getHref(),
      'target_path' => $this->getPath(),
      'target_scheme' => $this->getScheme(),
      'target_entity_type' => $this->getEntityType(),
      'target_entity_id' => $this->getEntityId(),
      'target_entity_langcode' => $this->getEntityLangcode(),
      'target_hash_id' => $this->getHashId(),
      'target_entity_bundle' => $this->getEntityBundle(),
      'target_title' => $this->getTitle(),
    ];
  }

}
