<?php

namespace Drupal\Tests\entity_mesh\Kernel;

use Drupal\Core\Session\AccountInterface;
use Drupal\file\Entity\File;
use Drupal\filter\Entity\FilterFormat;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\MediaTypeInterface;
use Drupal\media\Entity\Media;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\entity_mesh\Kernel\Traits\EntityMeshTestTrait;
use Drupal\user\Entity\Role;
use org\bovigo\vfs\vfsStream;

/**
 * Entity mesh kernel base.
 */
abstract class EntityMeshTestBase extends KernelTestBase {

  use ContentTypeCreationTrait;
  use MediaTypeCreationTrait;
  use UserCreationTrait;
  use EntityMeshTestTrait;

  /**
   * Modules to enable.
   *
   * @var array<string>
   */
  protected static $modules = [
    'system',
    'node',
    'user',
    'field',
    'filter',
    'text',
    'language',
    'entity_mesh',
    'path_alias',
    'file',
    'image',
    'media',
  ];

  /**
   * Provider defaults for test cases.
   *
   * @var array<string>
   */
  protected static array $providerDefaults = [
    'source_entity_id'                 => NULL,
    'source_entity_langcode'           => NULL,
    'target_href'                      => NULL,
    'expected_target_link_type'        => NULL,
    'expected_target_subcategory'      => NULL,
    'expected_target_title'            => NULL,
    'expected_target_scheme'           => NULL,
    'expected_target_entity_type'      => NULL,
    'expected_target_entity_bundle'    => NULL,
    'expected_target_entity_id'        => NULL,
    'expected_target_entity_langcode'  => NULL,
    'expected_source_entity_type'      => NULL,
    'expected_source_entity_bundle'    => NULL,
    'expected_source_entity_langcode'  => NULL,
    'expected_source_title'            => NULL,
    'source_should_exist'              => TRUE,
  ];

  /**
   * File Media Type.
   *
   * @var \Drupal\media\MediaTypeInterface
   */
  protected $fileMediaType;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install the necessary schemas.
    $this->installEntitySchema('configurable_language');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installEntitySchema('path_alias');
    $this->installSchema('file', 'file_usage');
    $this->installSchema('entity_mesh', ['entity_mesh']);
    $this->installConfig(['filter', 'node', 'file', 'media', 'system', 'language', 'entity_mesh']);
    $this->installSchema('node', ['node_access']);

    $this->createContentType(['type' => 'page', 'name' => 'Page']);

    // Create a file media type.
    $this->fileMediaType = $this->createMediaType('file', ['id' => 'document', 'label' => 'Document']);

    $config = $this->config('language.negotiation');
    $config->set('url.prefixes', ['en' => 'en'])
      ->save();

    // Enable the body field in the default view mode.
    $this->container->get('entity_display.repository')
      ->getViewDisplay('node', 'page', 'full')
      ->setComponent('body', [
        // Show label above the body content.
        'label' => 'above',
        // Render as basic text.
        'type' => 'text_default',
      ])
      ->save();

    $filter_format = FilterFormat::load('basic_html');
    if (!$filter_format) {
      $filter_format = FilterFormat::create([
        'format' => 'basic_html',
        'name' => 'Basic HTML',
        'filters' => [],
      ]);
      $filter_format->save();
    }

    if (!Role::load(AccountInterface::ANONYMOUS_ROLE)) {
      Role::create(['id' => AccountInterface::ANONYMOUS_ROLE, 'label' => 'Anonymous user'])->save();
    }

    $this->grantPermissions(Role::load(AccountInterface::ANONYMOUS_ROLE), [
      'access content',
      'view media',
    ]);

    // Set entity_mesh to synchronous mode by default for tests.
    // This ensures backward compatibility with existing tests that expect
    // immediate processing.
    $config = $this->config('entity_mesh.settings');
    $config->set('processing_mode', 'synchronous');
    // High limit to process everything.
    $config->set('synchronous_limit', 9999);
    $config->save();

  }

  /**
   * Creates a node with given content.
   */
  abstract protected function createExampleNodes();

  /**
   * Provides test cases for different types of links.
   */
  abstract public static function linkCasesProvider();

  /**
   * Fetches records from the 'entity_mesh' table.
   */
  protected function fetchEntityMeshRecords() {

    $connection = $this->container->get('database');
    $query = $connection->select('entity_mesh', 'em')
      ->fields('em', [
        'id',
        'type',
        'category',
        'subcategory',
        'source_entity_id',
        'source_entity_type',
        'source_entity_bundle',
        'source_entity_langcode',
        'source_title',
        'target_href',
        'target_path',
        'target_scheme',
        'target_link_type',
        'target_entity_type',
        'target_entity_bundle',
        'target_entity_id',
        'target_title',
        'target_entity_langcode',
      ]);
    $result = $query->execute();
    return $result ? $result->fetchAllAssoc('id', \PDO::FETCH_ASSOC) : [];
  }

  /**
   * Tests different types of links.
   *
   * @dataProvider linkCasesProvider
   */
  public function testLinks(
    $source_entity_id,
    $source_entity_langcode,
    $target_href,
    $expected_target_link_type,
    $expected_target_subcategory = NULL,
    $expected_target_title = NULL,
    $expected_target_scheme = NULL,
    $expected_target_entity_type = NULL,
    $expected_target_entity_bundle = NULL,
    $expected_target_entity_id = NULL,
    $expected_target_entity_langcode = NULL,
    $expected_source_entity_type = NULL,
    $expected_source_entity_bundle = NULL,
    $expected_source_entity_langcode = NULL,
    $expected_source_title = NULL,
    $source_should_exist = TRUE,
  ) {

    // Fetch records from entity_mesh table for assertions.
    $records = $this->fetchEntityMeshRecords();

    // Filter records based on node ID and link type.
    $filtered = array_filter($records, function ($record) use ($source_entity_id, $target_href, $source_entity_langcode) {
      $matches = $record['source_entity_id'] == $source_entity_id &&
        ($record['target_href'] === $target_href || ($record['target_entity_type'] === 'media' && str_contains($record['target_href'], $target_href)));

      if ($source_entity_langcode !== NULL) {
        $matches = $matches && $record['source_entity_langcode'] === $source_entity_langcode;
      }

      return $matches;
    });

    // Extract the record by matching 'target_href' with $expected_href.
    $record = reset($filtered);

    if ($source_should_exist) {
      $this->assertNotEmpty($record, "No record found with source_entity_id: $source_entity_id and target_href: $target_href");
    }
    else {
      $this->assertEmpty($record, "Record found with source_entity_id: $source_entity_id and target_href: $target_href");
      return;
    }

    $checks = [
      'target_link_type'       => $expected_target_link_type,
      'subcategory'            => $expected_target_subcategory,
      'target_title'           => $expected_target_title,
      'target_scheme'          => $expected_target_scheme,
      'target_entity_type'     => $expected_target_entity_type,
      'target_entity_bundle'   => $expected_target_entity_bundle,
      'target_entity_id'       => $expected_target_entity_id,
      'target_entity_langcode' => $expected_target_entity_langcode,
      'source_entity_type'     => $expected_source_entity_type,
      'source_entity_bundle'   => $expected_source_entity_bundle,
      'source_entity_langcode' => $expected_source_entity_langcode,
      'source_title'           => $expected_source_title,
    ];

    foreach ($checks as $key => $expectedValue) {
      if ($expectedValue !== NULL) {
        $this->assertEquals($expectedValue, $record[$key]);
      }
    }
  }

  /**
   * Helper to generate a media item.
   *
   * @param string $filename
   *   String filename with extension.
   * @param \Drupal\media\MediaTypeInterface $media_type
   *   The media type.
   * @param int $mid
   *   The media ID.
   *
   * @return \Drupal\media\Entity\Media
   *   A media item.
   */
  protected function generateMedia($filename, MediaTypeInterface $media_type, int $mid) {
    vfsStream::setup('drupal_root');
    vfsStream::create([
      'sites' => [
        'default' => [
          'files' => [
            $filename => str_repeat('a', 3000),
          ],
        ],
      ],
    ]);

    $file = File::create([
      'uri' => 'public://' . $filename,
    ]);
    $file->setPermanent();
    $file->save();

    return Media::create([
      'bundle' => $media_type->id(),
      'name' => $filename,
      'mid' => $mid,
      'field_media_file' => [
        'target_id' => $file->id(),
      ],
    ]);
  }

  /**
   * Helper to the url from uri.
   *
   * @param string $mid
   *   Media ID.
   *
   * @return string
   *   The url.
   */
  public static function getUrlFromMedia($mid) {
    $media = Media::load($mid);
    $uri = $media->get('field_media_file')->entity->getFileUri();
    return \Drupal::service('file_url_generator')->generateString($uri);
  }

}
