<?php

namespace Drupal\entity_mesh;

use Drupal\Core\Session\AccountInterface;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * Dummy account class for check access to entities base on roles.
 */
final class DummyAccount implements AccountInterface {

  /**
   * Roles of the dummy account.
   *
   * @var array
   */
  protected $roles = [];

  /**
   * Constructs a DummyAccount object.
   */
  public static function create() {
    $instance = new static();
    return $instance;
  }

  /**
   * Set the dummy account as anonymous.
   */
  public function setAsAnonymous() {
    $this->roles = [AccountInterface::ANONYMOUS_ROLE];
  }

  /**
   * Set role for the dummy account.
   *
   * @param string $role
   *   Role.
   */
  public function setRole(string $role) {
    $this->roles[] = $role;
  }

  /**
   * {@inheritdoc}
   */
  public function id() {
    // Default ID.
    return 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getRoles($exclude_locked_roles = FALSE) {
    $roles = $this->roles;

    if ($exclude_locked_roles) {
      $roles = array_values(array_diff($roles, [AccountInterface::ANONYMOUS_ROLE, AccountInterface::AUTHENTICATED_ROLE]));
    }

    // Default empty roles array.
    return $roles;
  }

  /**
   * {@inheritdoc}
   */
  public function hasPermission($permission) {
    foreach ($this->roles as $role) {
      $role_entity = Role::load($role);
      if (!$role_entity instanceof RoleInterface) {
        continue;
      }
      if ($role_entity->hasPermission($permission)) {
        // If any role has the permission, return TRUE.
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function isAuthenticated() {
    return in_array(AccountInterface::AUTHENTICATED_ROLE, $this->roles);
  }

  /**
   * {@inheritdoc}
   */
  public function isAnonymous() {
    return in_array(AccountInterface::ANONYMOUS_ROLE, $this->roles);
  }

  /**
   * {@inheritdoc}
   */
  public function getPreferredLangcode($fallback_to_default = TRUE) {
    // Default language code.
    return 'en';
  }

  /**
   * {@inheritdoc}
   */
  public function getPreferredAdminLangcode($fallback_to_default = TRUE) {
    // Default admin language code.
    return 'en';
  }

  /**
   * {@inheritdoc}
   */
  public function getAccountName() {
    // Default account name.
    return 'dummy-account';
  }

  /**
   * {@inheritdoc}
   */
  public function getDisplayName() {
    // Default display name.
    return 'Dummy Account';
  }

  /**
   * {@inheritdoc}
   */
  public function getEmail() {
    // Default empty email.
    return "dummy-account@dummyaccount.com";
  }

  /**
   * {@inheritdoc}
   */
  public function getTimeZone() {
    // Default timezone.
    return 'UTC';
  }

  /**
   * {@inheritdoc}
   */
  public function getLastAccessedTime() {
    // Default last accessed time.
    return 0;
  }

}
