<?php

namespace Drupal\Tests\entity_mesh\Kernel;

use Drupal\Core\Session\AccountInterface;
use Drupal\filter\Entity\FilterFormat;
use Drupal\node\Entity\Node;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\user\Entity\Role;

/**
 * Tests entity_mesh permissions handling.
 *
 * @group entity_mesh
 */
class EntityMeshPermissionsTest extends KernelTestBase {

  use ContentTypeCreationTrait;
  use UserCreationTrait;

  /**
   * Modules to enable.
   *
   * @var array<string>
   */
  protected static $modules = [
    'system',
    'node',
    'user',
    'field',
    'filter',
    'text',
    'language',
    'entity_mesh',
    'path_alias',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install the necessary schemas.
    $this->installEntitySchema('configurable_language');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installEntitySchema('path_alias');
    $this->installSchema('entity_mesh', ['entity_mesh']);
    $this->installConfig(['filter', 'node', 'system', 'language', 'entity_mesh']);
    $this->installSchema('node', ['node_access']);

    $this->createContentType(['type' => 'page', 'name' => 'Page']);

    $config = $this->config('language.negotiation');
    $config->set('url.prefixes', ['en' => 'en'])
      ->save();

    // Enable the body field in the default view mode.
    $this->container->get('entity_display.repository')
      ->getViewDisplay('node', 'page', 'full')
      ->setComponent('body', [
        // Show label above the body content.
        'label' => 'above',
        // Render as basic text.
        'type' => 'text_default',
      ])
      ->save();

    $filter_format = FilterFormat::load('basic_html');
    if (!$filter_format) {
      $filter_format = FilterFormat::create([
        'format' => 'basic_html',
        'name' => 'Basic HTML',
        'filters' => [],
      ]);
      $filter_format->save();
    }

    if (!Role::load(AccountInterface::ANONYMOUS_ROLE)) {
      Role::create(['id' => AccountInterface::ANONYMOUS_ROLE, 'label' => 'Anonymous user'])->save();
    }

    $this->grantPermissions(Role::load(AccountInterface::ANONYMOUS_ROLE), [
      'access content',
    ]);
  }

  /**
   * Tests that entities without anonymous access are excluded from analysis.
   */
  public function testEntitiesWithoutAnonymousAccessAreExcluded() {

    $html_node = '
      <p>External link https schema: <a href="https://example.com">Example</a></p>
      <p>External link http schema: <a href="http://example.com">Example</a></p>
      <p>Internal broken link: <a href="/non-existent">Broken</a></p>
      <p>Iframe content: <iframe src="https://example.com/iframe"></iframe></p>
      <p>Mailto link: <a href="mailto:hola@metadrop.net">Mailto link</a></p>
      <p>Tel link: <a href="tel:+34654654654">Tel link</a></p>
    ';

    // Create an unpublished node that anonymous users cannot access.
    $restricted_node = Node::create([
      'type' => 'page',
      'title' => 'Unpublished Node (No Anonymous Access)',
    // Owner user ID.
      'uid' => 1,
    // Unpublished.
      'status' => 0,
      'body' => [
        'value' => $html_node,
        'format' => 'basic_html',
      ],
    ]);
    $restricted_node->save();

    // Create a published node that anonymous users can access.
    $public_node = Node::create([
      'type' => 'page',
      'title' => 'Published Node (Anonymous Access)',
    // Owner user ID.
      'uid' => 1,
    // Published.
      'status' => 1,
      'body' => [
        'value' => $html_node,
        'format' => 'basic_html',
      ],
    ]);

    $public_node->save();

    // Verify anonymous user can access public node but not restricted.
    $anonymous = $this->container->get('entity_type.manager')->getStorage('user')->load(0);

    // Confirm access permissions.
    $this->assertTrue($public_node->access('view', $anonymous), 'Anonymous user should have access to public node');
    $this->assertFalse($restricted_node->access('view', $anonymous), 'Anonymous user should NOT have access to restricted node');

    // Update nodes to trigger entity_mesh processing via hook_entity_update.
    // Entity mesh processes entities on insert/update/delete hooks.
    $restricted_node->setTitle('Updated: ' . $restricted_node->getTitle());
    $restricted_node->save();

    $public_node->setTitle('Updated: ' . $public_node->getTitle());
    $public_node->save();

    // Query the entity_mesh table to check which nodes were processed.
    $database = $this->container->get('database');

    // Get all sources from the repository.
    $query = $database->select('entity_mesh', 'em')
      ->fields('em', ['source_entity_type', 'source_entity_id']);
    $results = $query->execute()->fetchAll();

    // Convert results to a more usable format.
    $processed_entity_ids = [];
    foreach ($results as $result) {
      $processed_entity_ids[] = $result->source_entity_id;
    }

    // Assert that the public node was processed.
    $this->assertContains($public_node->id(), $processed_entity_ids, 'Public node should be included in entity_mesh analysis.');

    // Assert that the restricted node was NOT processed
    // (this test should fail initially).
    $this->assertNotContains($restricted_node->id(), $processed_entity_ids, 'Restricted node should be excluded from entity_mesh analysis.');
  }

}
