<?php

namespace Drupal\Tests\entity_mesh\Unit;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\entity_mesh\DummyAccount;
use Drupal\entity_mesh\Repository;
use Drupal\Tests\UnitTestCase;
use Drupal\user\UserInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Tests the Repository class.
 *
 * @group entity_mesh
 * @coversDefaultClass \Drupal\entity_mesh\Repository
 */
class RepositoryTest extends UnitTestCase {

  /**
   * The repository service under test.
   *
   * @var \Drupal\entity_mesh\Repository
   */
  protected $repository;

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The mocked request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $requestStack;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->requestStack = $this->createMock(RequestStack::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityFieldManager = $this->createMock(EntityFieldManagerInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);

    // Mock the language.negotiation config for the constructor.
    $languageConfig = $this->createMock(Config::class);
    $languageConfig->expects($this->any())
      ->method('get')
      ->with('url.prefixes')
      ->willReturn([]);

    $this->configFactory->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['language.negotiation', $languageConfig],
      ]);

    $this->repository = new Repository(
      $this->database,
      $this->logger,
      $this->requestStack,
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->configFactory
    );
  }

  /**
   * Tests getMeshAccount with anonymous account type.
   *
   * @covers ::getMeshAccount
   */
  public function testGetMeshAccountWithAnonymousType() {
    // Mock entity_mesh.settings config.
    $settingsConfig = $this->createMock(Config::class);
    $settingsConfig->expects($this->once())
      ->method('get')
      ->with('analyzer_account')
      ->willReturn(['type' => 'anonymous', 'roles' => NULL, 'user_id' => NULL]);

    // Mock language.negotiation config for constructor.
    $languageConfig = $this->createMock(Config::class);
    $languageConfig->expects($this->any())
      ->method('get')
      ->with('url.prefixes')
      ->willReturn([]);

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['entity_mesh.settings', $settingsConfig],
        ['language.negotiation', $languageConfig],
      ]);

    // Create a new repository instance with the mocked config factory.
    $repository = new Repository(
      $this->database,
      $this->logger,
      $this->requestStack,
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->configFactory
    );

    $account = $repository->getMeshAccount();

    $this->assertInstanceOf(DummyAccount::class, $account);
    $this->assertTrue($account->isAnonymous());
    $this->assertContains('anonymous', $account->getRoles());
  }

  /**
   * Tests getMeshAccount with authenticated account type.
   *
   * @covers ::getMeshAccount
   */
  public function testGetMeshAccountWithAuthenticatedType() {
    $additionalRoles = ['editor', 'administrator'];

    // Mock entity_mesh.settings config.
    $settingsConfig = $this->createMock(Config::class);
    $settingsConfig->expects($this->once())
      ->method('get')
      ->with('analyzer_account')
      ->willReturn(['type' => 'authenticated', 'roles' => $additionalRoles, 'user_id' => NULL]);

    // Mock language.negotiation config for constructor.
    $languageConfig = $this->createMock(Config::class);
    $languageConfig->expects($this->any())
      ->method('get')
      ->with('url.prefixes')
      ->willReturn([]);

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['entity_mesh.settings', $settingsConfig],
        ['language.negotiation', $languageConfig],
      ]);

    // Create a new repository instance with the mocked config factory.
    $repository = new Repository(
      $this->database,
      $this->logger,
      $this->requestStack,
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->configFactory
    );

    $account = $repository->getMeshAccount();

    $this->assertInstanceOf(DummyAccount::class, $account);
    $this->assertTrue($account->isAuthenticated());
    $this->assertFalse($account->isAnonymous());
    $expectedRoles = ['authenticated', 'editor', 'administrator'];
    $this->assertEquals($expectedRoles, $account->getRoles());
  }

  /**
   * Tests that getMeshAccount uses instance caching.
   *
   * @covers ::getMeshAccount
   */
  public function testGetMeshAccountCaching() {
    // Mock entity_mesh.settings config.
    $settingsConfig = $this->createMock(Config::class);
    // Expect get() to be called only once due to instance caching.
    $settingsConfig->expects($this->once())
      ->method('get')
      ->with('analyzer_account')
      ->willReturn(['type' => 'authenticated', 'roles' => [], 'user_id' => NULL]);

    // Mock language.negotiation config for constructor.
    $languageConfig = $this->createMock(Config::class);
    $languageConfig->expects($this->any())
      ->method('get')
      ->with('url.prefixes')
      ->willReturn([]);

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['entity_mesh.settings', $settingsConfig],
        ['language.negotiation', $languageConfig],
      ]);

    // Create a new repository instance with the mocked config factory.
    $repository = new Repository(
      $this->database,
      $this->logger,
      $this->requestStack,
      $this->entityTypeManager,
      $this->entityFieldManager,
      $this->configFactory
    );

    // Call getMeshAccount multiple times.
    $account1 = $repository->getMeshAccount();
    $account2 = $repository->getMeshAccount();
    $account3 = $repository->getMeshAccount();

    // All calls should return the same instance due to instance caching.
    $this->assertSame($account1, $account2);
    $this->assertSame($account2, $account3);
  }

  /**
   * Tests getMeshAccount with user account type.
   *
   * @covers ::getMeshAccount
   */
  public function testGetMeshAccountWithUserType() {
    $userId = 123;
    $userRoles = ['authenticated', 'editor'];

    // Mock user entity.
    $user = $this->createMock(UserInterface::class);
    $user->expects($this->once())
      ->method('getRoles')
      ->willReturn($userRoles);

    // Mock user storage.
    $userStorage = $this->createMock(EntityStorageInterface::class);
    $userStorage->expects($this->once())
      ->method('load')
      ->with($userId)
      ->willReturn($user);

    // Mock entity type manager.
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->expects($this->once())
      ->method('getStorage')
      ->with('user')
      ->willReturn($userStorage);

    // Mock entity_mesh.settings config.
    $settingsConfig = $this->createMock(Config::class);
    $settingsConfig->expects($this->once())
      ->method('get')
      ->with('analyzer_account')
      ->willReturn(['type' => 'user', 'roles' => NULL, 'user_id' => $userId]);

    // Mock language.negotiation config for constructor.
    $languageConfig = $this->createMock(Config::class);
    $languageConfig->expects($this->any())
      ->method('get')
      ->with('url.prefixes')
      ->willReturn([]);

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['entity_mesh.settings', $settingsConfig],
        ['language.negotiation', $languageConfig],
      ]);

    // Create a new repository instance with the mocked dependencies.
    $repository = new Repository(
      $this->database,
      $this->logger,
      $this->requestStack,
      $entityTypeManager,
      $this->entityFieldManager,
      $this->configFactory
    );

    $account = $repository->getMeshAccount();

    $this->assertInstanceOf(DummyAccount::class, $account);
    $this->assertTrue($account->isAuthenticated());
    $this->assertFalse($account->isAnonymous());
    $this->assertEquals($userRoles, $account->getRoles());
  }

}
