<?php

namespace Drupal\Tests\entity_mesh\Unit;

use Drupal\entity_mesh\Target;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Tests the Target class URL validation.
 *
 * @group entity_mesh
 * @coversDefaultClass \Drupal\entity_mesh\Target
 */
class TargetTest extends UnitTestCase {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a mock request stack with a request.
    $this->requestStack = new RequestStack();
    $request = Request::create('http://example.com');
    $this->requestStack->push($request);
  }

  /**
   * Creates a Target instance for testing.
   *
   * @return \Drupal\entity_mesh\Target
   *   The target instance.
   */
  protected function createTarget(): Target {
    return Target::create($this->requestStack);
  }

  /**
   * Tests that valid telephone links are processed.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testTelephoneLinkCategorization() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('tel:+34654654654');

    // Valid tel: links return validation code 2 and continue normal processing.
    $this->assertEquals('tel:+34654654654', $target->getHref());
    $this->assertEquals('tel', $target->getScheme());
  }

  /**
   * Tests that mailto links are processed.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testMailtoLinkCategorization() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('mailto:test@example.com');

    // Mailto links return validation code 2 and continue normal processing.
    $this->assertEquals('mailto:test@example.com', $target->getHref());
    $this->assertEquals('mailto', $target->getScheme());
  }

  /**
   * Tests that javascript links are processed.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testJavascriptLinkCategorization() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('javascript:void(0)');

    // Javascript links return validation code 2 and continue normal processing.
    $this->assertEquals('javascript:void(0)', $target->getHref());
    $this->assertEquals('javascript', $target->getScheme());
  }

  /**
   * Tests that data URIs are processed.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testDataUriCategorization() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('data:image/png;base64,iVBORw0KGgo=');

    // Data URIs return validation code 2 and continue normal processing.
    $this->assertEquals('data:image/png;base64,iVBORw0KGgo=', $target->getHref());
    $this->assertEquals('data', $target->getScheme());
  }

  /**
   * Tests that FTP links are processed.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testFtpLinkCategorization() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('ftp://ftp.example.com/file.txt');

    // FTP links return validation code 2 and continue normal processing.
    $this->assertEquals('ftp://ftp.example.com/file.txt', $target->getHref());
    $this->assertEquals('ftp', $target->getScheme());
  }

  /**
   * Tests that valid HTTP URLs are processed correctly.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testHttpUrlValidation() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('http://external.com/page');

    $this->assertEquals('external', $target->getLinkType());
    $this->assertEquals('http', $target->getScheme());
    $this->assertEquals('external.com', $target->getHost());
  }

  /**
   * Tests that valid HTTPS URLs are processed correctly.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testHttpsUrlValidation() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('https://external.com/page');

    $this->assertEquals('external', $target->getLinkType());
    $this->assertEquals('https', $target->getScheme());
    $this->assertEquals('external.com', $target->getHost());
  }

  /**
   * Tests that relative internal URLs are processed correctly.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testRelativeInternalUrlValidation() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('/node/123');

    $this->assertEquals('internal', $target->getLinkType());
    $this->assertEquals('/node/123', $target->getPath());
  }

  /**
   * Tests that empty URLs are handled correctly.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testEmptyUrlValidation() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('   ');

    // Empty URLs should be categorized as invalid.
    $this->assertEquals('link', $target->getCategory());
    $this->assertEquals('invalid-url', $target->getSubcategory());
  }

  /**
   * Tests that malformed URLs are categorized as invalid.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testMalformedUrlCategorization() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('ht!tp://invalid url with spaces');

    $this->assertEquals('link', $target->getCategory());
    $this->assertEquals('invalid-url', $target->getSubcategory());
  }

  /**
   * Tests case-insensitive scheme detection.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testCaseInsensitiveSchemeDetection() {
    // Test uppercase TEL - href is preserved as-is, but validation works.
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('TEL:+1234567890');

    $this->assertEquals('TEL:+1234567890', $target->getHref());
    // Scheme from parse_url is lowercase.
    $this->assertNotNull($target->getScheme());

    // Test mixed case MailTo.
    $target2 = $this->createTarget();
    $target2->processHrefAndSetComponents('MailTo:user@example.com');

    $this->assertEquals('MailTo:user@example.com', $target2->getHref());
    $this->assertNotNull($target2->getScheme());
  }

  /**
   * Tests URL decoding before validation.
   *
   * @covers ::processHrefAndSetComponents
   */
  public function testUrlDecoding() {
    $target = $this->createTarget();
    $encoded_url = 'https://example.com/path%20with%20spaces';
    $target->processHrefAndSetComponents($encoded_url);

    // The href should be stored decoded.
    $this->assertStringContainsString('path with spaces', $target->getHref());
  }

  /**
   * Tests that empty tel: links are marked as invalid.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testEmptyTelephoneValidation() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('tel:');

    $this->assertEquals('link', $target->getCategory());
    $this->assertEquals('invalid-tel', $target->getSubcategory());
  }

  /**
   * Tests that tel: links with no digits are marked as invalid.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testTelephoneWithNoDigits() {
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('tel:abc');

    $this->assertEquals('link', $target->getCategory());
    $this->assertEquals('invalid-tel', $target->getSubcategory());
  }

  /**
   * Tests that tel: links with invalid characters are marked as invalid.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testTelephoneWithInvalidCharacters() {
    // Test with script tag.
    $target = $this->createTarget();
    $target->processHrefAndSetComponents('tel:+34<script>alert(1)</script>');

    $this->assertEquals('link', $target->getCategory());
    $this->assertEquals('invalid-tel', $target->getSubcategory());

    // Test with spaces (not allowed in new validation).
    $target2 = $this->createTarget();
    $target2->processHrefAndSetComponents('tel:+1 234 567 8901');

    $this->assertEquals('link', $target2->getCategory());
    $this->assertEquals('invalid-tel', $target2->getSubcategory());

    // Test with parentheses (not allowed in new validation).
    $target3 = $this->createTarget();
    $target3->processHrefAndSetComponents('tel:+1(234)567-8901');

    $this->assertEquals('link', $target3->getCategory());
    $this->assertEquals('invalid-tel', $target3->getSubcategory());
  }

  /**
   * Tests that valid tel: links with various formats are accepted.
   *
   * New validation only allows: optional +, digits, dots, and hyphens.
   *
   * @covers ::processHrefAndSetComponents
   * @covers ::isValidUrl
   */
  public function testValidTelephoneFormats() {
    // Test E.164 format with plus.
    $target1 = $this->createTarget();
    $target1->processHrefAndSetComponents('tel:+12345678901');
    $this->assertEquals('tel:+12345678901', $target1->getHref());
    $this->assertNotEquals('invalid-tel', $target1->getSubcategory());

    // Test with dashes.
    $target2 = $this->createTarget();
    $target2->processHrefAndSetComponents('tel:+1-234-567-8901');
    $this->assertEquals('tel:+1-234-567-8901', $target2->getHref());
    $this->assertNotEquals('invalid-tel', $target2->getSubcategory());

    // Test with dots.
    $target3 = $this->createTarget();
    $target3->processHrefAndSetComponents('tel:+1.234.567.8901');
    $this->assertEquals('tel:+1.234.567.8901', $target3->getHref());
    $this->assertNotEquals('invalid-tel', $target3->getSubcategory());

    // Test without plus.
    $target4 = $this->createTarget();
    $target4->processHrefAndSetComponents('tel:1234567890');
    $this->assertEquals('tel:1234567890', $target4->getHref());
    $this->assertNotEquals('invalid-tel', $target4->getSubcategory());

    // Test mixed dots and dashes.
    $target5 = $this->createTarget();
    $target5->processHrefAndSetComponents('tel:+34.654-654-654');
    $this->assertEquals('tel:+34.654-654-654', $target5->getHref());
    $this->assertNotEquals('invalid-tel', $target5->getSubcategory());
  }

}
