<?php

namespace Drupal\entity_note_block\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\entity_note_block\Service\NoteStorage;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form class for displaying and managing entity notes via a modal block.
 */
class AddNoteForm extends FormBase {

  /**
   * The note storage service.
   *
   * @var \Drupal\entity_note_block\Service\NoteStorage
   */
  protected $noteStorage;

  /**
   * Constructs the form with dependency injection.
   *
   * @param \Drupal\entity_note_block\Service\NoteStorage $noteStorage
   *   The note storage service.
   */
  public function __construct(NoteStorage $noteStorage) {
    $this->noteStorage = $noteStorage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_note_block.storage')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'add_entity_note_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#prefix'] = '<div id="entity-note-form-wrapper">';
    $form['#suffix'] = '</div>';

    // Load notes to display in the table.
    $notes = $this->noteStorage->getNotes();

    // Build the notes table.
    $form['notes_table'] = [
      '#type' => 'table',
      '#header' => [$this->t('Date'), $this->t('Note'), $this->t('Edit'), $this->t('Delete')],
      '#rows' => [],
      '#attributes' => ['id' => 'note-table'],
      '#empty' => $this->t('No notes yet.'),
    ];

    foreach ($notes as $index => $note) {
      $form['notes_table']['row_' . $index] = [
        'date' => ['data' => ['#markup' => date('Y-m-d H:i', $note->created)]],
        'note' => ['data' => ['#markup' => $note->note]],
        'edit' => [
          'data' => [
            '#type' => 'submit',
            '#value' => $this->t('Edit'),
            '#name' => 'edit_' . $note->id,
            '#ajax' => [
              'callback' => '::loadNote',
              'wrapper' => 'entity-note-form-wrapper',
            ],
          ],
        ],
        'delete' => [
          'data' => [
            '#type' => 'submit',
            '#value' => $this->t('Delete'),
            '#name' => 'delete_' . $note->id,
            '#ajax' => [
              'callback' => '::deleteNote',
              'wrapper' => 'entity-note-form-wrapper',
            ],
            '#attributes' => [
              'class' => ['delete-note-btn'],
            ],
          ],
        ],
      ];
    }

    // Textarea for new or edited note.
    $form['note'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Note'),
      '#rows' => 4,
      '#attributes' => ['id' => 'edit-note'],
      '#default_value' => $form_state->get('edit_note') ?? '',
    ];

    // Hidden field to store entity ID when editing.
    $form['edit_entity_id'] = [
      '#type' => 'hidden',
      '#attributes' => ['id' => 'edit-entity-id'],
      '#default_value' => $form_state->get('edit_entity_id') ?? '',
    ];

    // Action buttons (save, clear, reload, close).
    $form['actions'] = ['#type' => 'actions'];

    $form['actions']['save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#ajax' => [
        'callback' => '::saveAjax',
        'wrapper' => 'entity-note-form-wrapper',
      ],
    ];

    $form['actions']['clear'] = [
      '#type' => 'button',
      '#value' => $this->t('Clear'),
      '#ajax' => ['callback' => '::clearAjax'],
    ];

    $form['actions']['close'] = [
      '#type' => 'button',
      '#value' => $this->t('Close'),
    ];

    $form['actions']['reload'] = [
      '#type' => 'button',
      '#value' => $this->t('Reload'),
      '#attributes' => [
        'class' => ['reload-button-note-form'],
      ],
      '#ajax' => [
        'callback' => '::reloadAjax',
        'wrapper' => 'entity-note-form-wrapper',
      ],
    ];

    // Attach libraries.
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    $form['#attached']['library'][] = 'entity_note_block/note_form_styles';

    return $form;
  }

  /**
   * AJAX callback to reload the form.
   */
  public function reloadAjax(array &$form, FormStateInterface $form_state) {
    return $form;
  }

  /**
   * AJAX callback to save or update a note.
   */
  public function saveAjax(array &$form, FormStateInterface $form_state) {
    $note = $form_state->getValue('note');
    $entity_id = $form_state->getValue('edit_entity_id');

    // If entity ID exists, update note; otherwise insert a new one.
    if (!empty($entity_id)) {
      $this->noteStorage->updateNote($entity_id, $note);
    }
    else {
      $entity_id = $this->noteStorage->saveNoteWithNode($note);
      $this->noteStorage->saveNote($entity_id, $note);
    }

    // Build AJAX response to clear fields and reload form.
    $response = new AjaxResponse();
    $response->addCommand(new InvokeCommand('#edit-note', 'val', ['']));
    $response->addCommand(new InvokeCommand('#edit-entity-id', 'val', ['']));
    $response->addCommand(new InvokeCommand('.reload-button-note-form', 'trigger', ['click']));

    return $response;
  }

  /**
   * AJAX callback to clear the note fields.
   */
  public function clearAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new InvokeCommand('#edit-note', 'val', ['']));
    $response->addCommand(new InvokeCommand('#edit-entity-id', 'val', ['']));
    return $response;
  }

  /**
   * AJAX callback to populate the form with existing note data.
   */
  public function loadNote(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement()['#name'];
    $index = str_replace('edit_', '', $trigger);
    $all_values = $this->noteStorage->getNotesValue($index);

    $response = new AjaxResponse();
    $response->addCommand(new InvokeCommand('#edit-note', 'val', [$all_values->note]));
    $response->addCommand(new InvokeCommand('#edit-entity-id', 'val', [$all_values->entity_id]));
    return $response;
  }

  /**
   * AJAX callback to delete a note and refresh the form.
   */
  public function deleteNote(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement()['#name'];
    $entity_id = str_replace('delete_', '', $trigger);

    if (is_numeric($entity_id)) {
      $this->noteStorage->deleteNote((int) $entity_id);
    }

    $response = new AjaxResponse();
    $response->addCommand(new InvokeCommand('.reload-button-note-form', 'trigger', ['click']));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {}
}
