<?php

namespace Drupal\entity_note_block\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;

/**
 * Provides a block that displays a button to open the notes modal.
 *
 * @Block(
 *   id = "entity_note_modal_button_block",
 *   admin_label = @Translation("Entity Note Block"),
 * )
 */
class EntityNoteBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs the EntityNoteBlock block.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $route_match) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Generate a modal link to the notes form route.
    $url = Url::fromRoute('entity_note_block.notes_form');
    $url->setOption('attributes', [
      'class' => ['use-ajax', 'button', 'button--primary'],
      'data-dialog-type' => 'modal',
      'data-dialog-options' => json_encode(['width' => 600]),
    ]);

    // Build a renderable link element.
    $link = Link::fromTextAndUrl($this->t('📝 Add/View Notes'), $url)->toRenderable();

    // Attach the core AJAX dialog library.
    $link['#attached']['library'][] = 'core/drupal.dialog.ajax';

    return $link;
  }

}
