<?php

namespace Drupal\entity_note_block\Service;

use Drupal\Core\Database\Connection;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Service class for managing entity notes.
 */
class NoteStorage {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $db;

  /**
   * Constructs the NoteStorage service.
   *
   * @param \Drupal\Core\Database\Connection $db
   *   The database connection.
   */
  public function __construct(Connection $db) {
    $this->db = $db;
  }

  /**
   * Retrieves all notes from the custom note table.
   *
   * @return array
   *   An array of note objects.
   */
  public function getNotes() {
    return $this->db->select('entity_note_block', 'n')
      ->fields('n')
      ->execute()
      ->fetchAll();
  }

  /**
   * Updates an existing note in the custom note table.
   *
   * @param int $entity_id
   *   The entity ID associated with the note.
   * @param string $note
   *   The updated note content.
   */
  public function updateNote($entity_id, $note) {
    $this->db->update('entity_note_block')
      ->fields([
        'note' => $note,
        'created' => \Drupal::time()->getRequestTime(),
      ])
      ->condition('entity_id', $entity_id)
      ->execute();
  }

  /**
   * Retrieves a single note by its entity ID.
   *
   * @param int $entity_id
   *   The entity ID associated with the note.
   *
   * @return object|false
   *   The note object or FALSE if not found.
   */
  public function getNotesValue($entity_id) {
    return $this->db->select('entity_note_block', 'n')
      ->fields('n')
      ->condition('entity_id', $entity_id)
      ->execute()
      ->fetchObject();
  }

  /**
   * Saves a note entry to the custom note table.
   *
   * @param int $entity_id
   *   The entity ID to associate the note with.
   * @param string $note
   *   The note content.
   */
  public function saveNote($entity_id, $note) {
    $this->db->insert('entity_note_block')
      ->fields([
        'entity_type' => 'entity_note_block',
        'entity_id' => $entity_id,
        'note' => $note,
        'created' => \Drupal::time()->getRequestTime(),
      ])
      ->execute();
  }

  /**
   * Creates a new node of type 'note_log' to store the note.
   *
   * If the content type or field does not exist, they are created.
   *
   * @param string $note
   *   The note content to store in the node.
   *
   * @return int
   *   The node ID of the created note.
   */
  public function saveNoteWithNode(string $note): int {
    $type = 'note_log';

    // Ensure the content type exists.
    if (!NodeType::load($type)) {
      $node_type = NodeType::create([
        'type' => $type,
        'name' => 'Note Log',
      ]);
      $node_type->save();

      // Create field storage if not exists.
      if (!FieldStorageConfig::loadByName('node', 'field_entity_notes')) {
        FieldStorageConfig::create([
          'field_name' => 'field_entity_notes',
          'entity_type' => 'node',
          'type' => 'text_long',
        ])->save();
      }

      // Attach field to content type if not exists.
      if (!FieldConfig::loadByName('node', $type, 'field_entity_notes')) {
        FieldConfig::create([
          'field_name' => 'field_entity_notes',
          'entity_type' => 'node',
          'bundle' => $type,
          'label' => 'Entity Notes',
        ])->save();
      }
    }

    // Create and save the node.
    $node = Node::create([
      'type' => $type,
      'title' => 'Note for entity',
      'field_entity_notes' => $note,
      'status' => 0,
    ]);
    $node->save();

    return $node->id() ?? 0;
  }

  /**
   * Deletes a note record and its associated node (if exists).
   *
   * @param int $entity_id
   *   The entity ID associated with the note and node.
   */
  public function deleteNote($entity_id) {
    // Delete from the custom notes table.
    $this->db->delete('entity_note_block')
      ->condition('entity_id', $entity_id)
      ->execute();

    // Delete the associated node if it exists.
    $node = Node::load($entity_id);
    if ($node && $node->bundle() === 'note_log') {
      $node->delete();
    }
  }

}
