<?php

namespace Drupal\entity_notes\Form;

use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\entity_notes\Entity\EntityNote;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * The Entity Notes page.
 */
class EntityNoteForm extends FormBase {

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * This note entity.
   *
   * @var \Drupal\entity_notes\Entity\EntityNoteInterface
   */
  protected $note;

  /**
   * The entity these notes are for.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * EntityNoteForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, RouteMatchInterface $route_match, EntityTypeManagerInterface $entity_type_manager) {
    $this->configFactory = $config_factory;
    $this->routeMatch = $route_match;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entity_note_form';
  }

  /**
   * Get entity from route.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getEntityFromRouteParameters() {
    $entity_type_id = $this->routeMatch->getRouteObject()->compile()->getVariables();
    $entity_id = $this->routeMatch->getParameter($entity_type_id[0]);
    $this->entity = $this->entityTypeManager->getStorage($entity_type_id[0])->load($entity_id);
  }

  /**
   * Title callback.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The page title.
   */
  public function title() {
    $this->getEntityFromRouteParameters();
    return $this->t('@label notes', ['@label' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $this->getEntityFromRouteParameters();
    $form_state->setTemporaryValue('entity', $this->entity);

    $form['notes'] = [
      'title' => [
        '#markup' => $this->t('<h2>Notes</h2>')
      ],
      'view' => [
        '#type' => 'view',
        '#name' => 'entity_notes',
        '#display_id' => 'entity_notes_block',
        '#arguments' => [
          $this->entity->id(),
          $this->entity->getEntityTypeId(),
        ],
      ],
      '#weight' => -10,
    ];

    // Attach fields from the entity note form display.
    $this->note = EntityNote::create([
      'entity_id' => $this->entity->id(),
      'entity_type_id' => $this->entity->getEntityTypeId(),
    ]);
    $form_display = EntityFormDisplay::collectRenderDisplay($this->note, 'edit');
    $form_state->set('form_display', $form_display);
    $form_display->buildForm($this->note, $form, $form_state);

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add new note'),
      '#weight' => 10,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Set the values, copied from ContentEntityForm::copyFormValuesToEntity().
    /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface $form_display */
    $form_display = $form_state->get('form_display');
    $extracted = $form_display->extractFormValues($this->note, $form, $form_state);
    foreach ($form_state->getValues() as $name => $values) {
      if ($this->note->hasField($name) && !isset($extracted[$name])) {
        $this->note->set($name, $values);
      }
    }

    $this->note->save();
  }

}
