<?php

namespace Drupal\entity_notes\Routing;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Entity Note Routes.
 */
class EntityNotesRoutes implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The Entity Notes logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The Entity Notes config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * EntityNotesRoutes constructor.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->logger = $logger_factory->get('entity_notes');
    $this->entityTypeManager = $entity_type_manager;
    $this->config = $config_factory->get('entity_notes.settings');
  }

  /**
   * Route provider callback.
   *
   * Adds a route for each entity that Entity Notes is enabled on.
   *
   * @return \Symfony\Component\Routing\RouteCollection
   *   The entity notes route collection.
   */
  public function routes() {
    $route_collection = new RouteCollection();
    $entity_types = $this->config->get('entity_types');

    if ($entity_types) {
      foreach ($entity_types as $entity_type_id) {

        try {
          $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
        }
        catch (\Exception $e) {
          // When installing a site from config the entity modules may not be
          // enabled yet.
          $this->logger->info("Skipped adding an entity_notes route for $entity_type_id. The module probably isn't enabled yet.");
          continue;
        }

        $canonical_path = $entity_type->getLinkTemplate('canonical');
        $route = new Route(
          "$canonical_path/notes",
          [
            '_form' => '\Drupal\entity_notes\Form\EntityNoteForm',
            '_title_callback' => '\Drupal\entity_notes\Form\EntityNoteForm::title',
          ],
          [
            '_permission' => 'add entity note entities',
          ]
        );
        $route_collection->add("entity_notes.$entity_type_id", $route);
      }
    }

    return $route_collection;
  }

}
