/**
 * @file
 * Initializes the entity reference tab formatter without jQuery UI.
 */

((Drupal, once) => {
    const activateTab = (wrapper, button, tabs, panels) => {
        tabs.forEach((tab) => {
            tab.setAttribute('aria-selected', 'false');
            tab.setAttribute('tabindex', '-1');
        });

        panels.forEach((panel) => {
            panel.hidden = true;
        });

        button.setAttribute('aria-selected', 'true');
        button.removeAttribute('tabindex');

        const targetId = button.getAttribute('data-entity-ref-target');
        if (targetId) {
            const target = wrapper.ownerDocument.getElementById(targetId);
            if (target) {
                target.hidden = false;
            }
        }
    };

    const focusTabByOffset = (currentIndex, offset, tabs) => {
        const newIndex = (currentIndex + offset + tabs.length) % tabs.length;
        return tabs[newIndex];
    };

    Drupal.behaviors.entityRefTabFormatter = {
        attach(context) {
            once('entity-ref-tab-formatter', '.entity-ref-tab-formatter-tabs', context).forEach((wrapper) => {
                const tabs = Array.from(wrapper.querySelectorAll('[data-entity-ref-tab]'));
                const panels = Array.from(wrapper.querySelectorAll('[data-entity-ref-panel]'));

                if (!tabs.length || !panels.length) {
                    return;
                }

                tabs.forEach((tab, index) => {
                    tab.setAttribute('aria-selected', 'false');
                    tab.setAttribute('tabindex', '-1');

                    tab.addEventListener('click', () => {
                        activateTab(wrapper, tab, tabs, panels);
                    });

                    tab.addEventListener('keydown', (event) => {
                        let targetTab;
                        switch (event.key) {
                            case 'ArrowRight':
                            case 'ArrowDown':
                                event.preventDefault();
                                targetTab = focusTabByOffset(index, 1, tabs);
                                break;

                            case 'ArrowLeft':
                            case 'ArrowUp':
                                event.preventDefault();
                                targetTab = focusTabByOffset(index, -1, tabs);
                                break;

                            case 'Home':
                                event.preventDefault();
                                targetTab = tabs[0];
                                break;

                            case 'End':
                                event.preventDefault();
                                targetTab = tabs[tabs.length - 1];
                                break;

                            default:
                                break;
                        }

                        if (targetTab) {
                            targetTab.focus();
                            activateTab(wrapper, targetTab, tabs, panels);
                        }
                    });

                    // Activate the first tab by default.
                    if (index === 0) {
                        activateTab(wrapper, tab, tabs, panels);
                        tab.removeAttribute('tabindex');
                    }
                });
            });
        },
    };
})(Drupal, once);
