<?php

namespace Drupal\entity_ref_tab_formatter\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element;
use Drupal\Field\FieldConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'entity_reference_tab_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "entity_reference_tab_formatter",
 *   label = @Translation("Entity reference tab formatter"),
 *   field_types = {
 *     "entity_reference",
 *     "entity_reference_revisions"
 *   }
 * )
 */
class EntityReferenceTabFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The bundle info service.
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * The entity display repository.
   */
  protected EntityDisplayRepositoryInterface $entityDisplayRepository;

  /**
   * Constructs a new formatter instance.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, EntityTypeBundleInfoInterface $bundle_info, EntityDisplayRepositoryInterface $entity_display_repository) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->bundleInfo = $bundle_info;
    $this->entityDisplayRepository = $entity_display_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_display.repository')
    );
  }
  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return parent::defaultSettings() + [
      'tab_title' => '',
      'tab_body' => '',
      'style' => 'tab',
      'rendered_view_mode' => 'default',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    // Implements the settings form.
    $field_settings = $this->getFieldSettings();
    $entity_type_id = $field_settings['target_type'] ?? NULL;
    $target_bundles = $field_settings['handler_settings']['target_bundles'] ?? [];
    if (empty($target_bundles) && $entity_type_id) {
      $target_bundles = array_keys($this->bundleInfo->getBundleInfo($entity_type_id));
    }

    $field_names = [];
    foreach ($target_bundles as $bundle) {
      $field_names = array_merge($field_names, array_keys($this->getEntityFields($entity_type_id, $bundle)));
    }
    $field_names = array_values(array_unique($field_names));
    sort($field_names, SORT_NATURAL | SORT_FLAG_CASE);

    $title_options = $field_names;
    if (!in_array('title', $title_options, TRUE)) {
      array_unshift($title_options, 'title');
    }
    $title_options = array_combine($title_options, $title_options);
    $body_options = $field_names ? array_combine($field_names, $field_names) : [];
    $body_options['__rendered_entity'] = $this->t('Rendered entity (full view)');

    $view_mode_options = [];
    if ($entity_type_id) {
      $target_bundle = reset($target_bundles) ?: NULL;
      $view_mode_options = $this->entityDisplayRepository->getViewModeOptionsByBundle($entity_type_id, $target_bundle) ?: [];
    }
    if (empty($view_mode_options)) {
      $view_mode_options = ['default' => $this->t('Default')];
    }

    $elements = [];
    if (!$entity_type_id || empty($title_options)) {
      $elements['notice'] = [
        '#markup' => $this->t('Configure target bundles with fields before selecting tab mappings.'),
        '#prefix' => '<p>',
        '#suffix' => '</p>',
      ];
      return $elements + parent::settingsForm($form, $form_state);
    }

    $elements['entity_type_id'] = [
      '#value' => $entity_type_id,
    ];
    $elements['tab_title'] = [
      '#type' => 'select',
      '#options' => $title_options,
      '#title' => $this->t('Select the tab title field.'),
      '#default_value' => $this->getSetting('tab_title') ?: 'title',
      '#required' => TRUE,
    ];
    $default_body = $this->getSetting('tab_body');
    if (!$default_body && $body_options) {
      $default_body = array_key_first($body_options);
    }
    $elements['tab_body'] = [
      '#type' => 'select',
      '#options' => $body_options,
      '#title' => $this->t('Select the tab body field.'),
      '#default_value' => $default_body,
      '#required' => TRUE,
    ];
    if (!$body_options) {
      $elements['tab_body']['#description'] = $this->t('No configurable fields were found on the selected bundle(s).');
      $elements['tab_body']['#empty_option'] = $this->t('- None available -');
      $elements['tab_body']['#required'] = FALSE;
      $elements['tab_body']['#disabled'] = TRUE;
    }

    $elements['rendered_view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Rendered entity view mode'),
      '#options' => $view_mode_options,
      '#default_value' => $this->getSetting('rendered_view_mode') ?: 'default',
      '#description' => $this->t('Applied when the tab body is set to "Rendered entity".'),
      '#states' => [
        'visible' => [
          sprintf(':input[name="fields[%s][settings_edit_form][settings][tab_body]"]', $this->fieldDefinition->getName()) => ['value' => '__rendered_entity'],
        ],
      ],
    ];
    $elements['style'] = [
      '#type' => 'radios',
      '#options' => [
        'tab' => $this->t('Tabs'),
        'accordion' => $this->t('Accordion'),
      ],
      '#title' => $this->t('Display style'),
      '#default_value' => $this->getSetting('style') ?: 'tab',
    ];

    return $elements + parent::settingsForm($form, $form_state);
  }

  /**
   * Helper function.
   */
  private function getEntityFields(?string $entity_type_id, ?string $bundle): array {
    if (empty($entity_type_id) || empty($bundle)) {
      return [];
    }

    $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
    return array_filter($fields, static fn($definition) => $definition instanceof FieldConfigInterface);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Title field: @field', ['@field' => $this->getSetting('tab_title') ?: $this->t('Not configured')]);
    if ($this->getSetting('tab_body') === '__rendered_entity') {
      $summary[] = $this->t('Body: Rendered entity (@view_mode view mode)', ['@view_mode' => $this->getSetting('rendered_view_mode')]);
    }
    else {
      $summary[] = $this->t('Body field: @field', ['@field' => $this->getSetting('tab_body') ?: $this->t('Not configured')]);
    }
    $summary[] = $this->t('Display style: @style', ['@style' => $this->getSetting('style') === 'accordion' ? $this->t('Accordion') : $this->t('Tabs')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $tabs = [];
    $title_field = $this->getSetting('tab_title');
    $body_field = $this->getSetting('tab_body');
    $use_rendered_entity = $body_field === '__rendered_entity';
    $style = $this->getSetting('style') ?: 'tab';
    $entity_type_id = $this->getFieldSettings()['target_type'] ?? NULL;

    if (!$entity_type_id || !$title_field || (!$use_rendered_entity && !$body_field)) {
      return $elements;
    }

    foreach ($items as $delta => $item) {
      $entity = $item->entity ?: ($item->target_id ? $this->entityTypeManager->getStorage($entity_type_id)->load($item->target_id) : NULL);
      if (!$entity) {
        continue;
      }

      $title = '';
      if ($title_field && $entity->hasField($title_field) && !$entity->get($title_field)->isEmpty()) {
        $title = $entity->get($title_field)->first()->getString();
      }
      elseif ($title_field === 'title' && method_exists($entity, 'label')) {
        $title = (string) $entity->label();
      }

      if ($title === '') {
        $title = $this->t('Item @number', ['@number' => $delta + 1]);
      }

      if ($use_rendered_entity) {
        $view_mode = $this->getSetting('rendered_view_mode') ?: 'default';
        $body_render = $this->entityTypeManager->getViewBuilder($entity->getEntityTypeId())->view($entity, $view_mode);
      }
      else {
        $body_render = ['#plain_text' => ''];
        if ($entity->hasField($body_field) && !$entity->get($body_field)->isEmpty()) {
          $body_render = $entity->get($body_field)->view($this->viewMode, $langcode);
          if (Element::isEmpty($body_render)) {
            // Fallback to the string value when the field cannot be rendered.
            $body_item = $entity->get($body_field)->first();
            if (isset($body_item->format)) {
              $body_render = [
                '#type' => 'processed_text',
                '#text' => $body_item->value,
                '#format' => $body_item->format,
              ];
            }
            else {
              $body_render = ['#plain_text' => $body_item->value];
            }
          }
        }
      }

      if (Element::isEmpty($body_render)) {
        $body_render = ['#markup' => $this->t('No content available.')];
      }

      $unique_id = sprintf('%s-%s', $entity->id(), $delta);
      $tabs[$unique_id] = [
        'title' => $title,
        'body' => $body_render,
      ];
    }

    if (!$tabs) {
      $elements[] = [
        '#markup' => $this->t('No referenced items available.'),
      ];
      return $elements;
    }

    $theme = 'entity_ref_tab_formatter';
    $library = 'entity_ref_tab_formatter/tab_formatter';
    if ($style === 'accordion') {
      $theme = 'entity_ref_accordion_formatter';
      $library = 'entity_ref_tab_formatter/accordion_formatter';
    }

    $build = [
      '#theme' => $theme,
      '#tabs' => $tabs,
    ];
    if ($library) {
      $build['#attached']['library'][] = $library;
    }

    $elements[] = $build;
    return $elements;
  }

  /**
   * Generate the output appropriate for one field item.
   *
   * @param \Drupal\Core\Field\FieldItemInterface $item
   *   One field item.
   *
   * @return string
   *   The textual output generated.
   */
  protected function viewValue(FieldItemInterface $item) {
    // The text value has no text format assigned to it, so the user input
    // should equal the output, including newlines.
    return nl2br(Html::escape($item->value));
  }

}
